#include <PalmOS.h>
#include "CountDownRcp.h"
#include "config.h"

#ifdef CONFIG_SONY
	#include <SonyCLIE.h>
#endif

#define appFileCreator						'Cntd'
#define appVersionNum              0x01
#define appPrefID                  0x00
#define appPrefVersionNum          0x02

#define ourMinVersion	sysMakeROMVersion(2,0,0,sysROMStageRelease,0)

#define noDate							0xFFFF
#define NoDateString				"no date"
#define NoAlarmString				"no alarm"
#define SelectDueDateTitle	"Select Due Date"
#define SelectAlarmTitle		"Select Alarm Time"

#define kOneNumberWidth			30
#define kOneNumberHeight		50
#define kNumberCornerDiam		4
#define kNumberSpacing			3
#define kNumberYOffset			45

#define kLinienAbstand			3
#define kLinienStaerke			3


/* FOR BEST FORMATING OF THIS FILE SET TABSTOP TO 2 */
/* THIS FILE WAS CREATED AND EDITED WITH VIM EDITOR */


// application structures -------------------------------------------------------------------------
typedef struct 
{
	DateType		due;
	TimeType		alarm;		// -1 if no time
	UInt16			reserved;
} CountDownPrefsType;

static UInt16 	RomVersionCompatible( UInt32 requiredVersion, UInt16 launchFlags );
static Boolean 	AppHandleEvent( EventType * eventP );
static void 		AppEventLoop( void );
static UInt16		AppStart( void );
static void 		AppStop( void );
static void			DisplayAlarm( SysDisplayAlarmParamType *cmdPBP );

static Boolean	MainFormHandleEvent( EventType * eventP );
static void			MainFormInit( FormType * frmP );
static void			MainFormSelectDueDate( ControlType * ctlP );
static void			MainFormSelectAlarm( ControlType * ctlP );
static void 		MainFormDrawDays( FormType * frmP );
static void			MainFormClearAlarm( FormType * frmP );
static void 		DrawDigit( Char digit, UInt8 x, UInt8 y );

static DateType				gDueDate;
static TimeType				gAlarmTime; // -1 if no time
static DateFormatType gPreferedDateFormat;
static TimeFormatType gPreferedTimeFormat;

#ifdef CONFIG_SONY
	static UInt16					gSonyLibRefNum;
#endif /* CONFIG_SONY */


/*
 * FUNCTION:			DrawDigit
 * DESCRIPTION:		draws a digit at the specified position
 * CALLED:				from MainFormDrawDays
 * PARAMETERS:		...
 * RETURNS:				nothing
 */
static void DrawDigit( Char digit, UInt8 x, UInt8 y )
{
	RectangleType r;
	UInt16				width, height;
	UInt16				mid;

	width = kLinienStaerke;
	mid = height = kOneNumberHeight/2;

	r.extent.x = width;
	r.extent.y = height;
	r.topLeft.x = x + (kOneNumberWidth - kLinienStaerke);

	// right up vert
	if( digit == '0' || digit == '1' || digit == '2' || digit == '3' || digit == '4' || digit == '7' || digit == '8' || digit == '9' )
	{
		r.topLeft.y = y;
		WinDrawRectangle( &r, kNumberCornerDiam );
	}
	// right down vert
	if( digit == '0' || digit == '1' || digit == '3' || digit == '4' || digit == '5' || digit == '6' || digit == '7' || digit == '8' || digit == '9' )
	{
		r.topLeft.y = y + height;
		WinDrawRectangle( &r, kNumberCornerDiam );
	}

	r.topLeft.x = x;

	// left up vert
	if( digit == '0' || digit == '4' || digit == '5' || digit == '6' || digit == '8' || digit == '9' )
	{
		r.topLeft.y = y;
		WinDrawRectangle( &r, kNumberCornerDiam );
	}
	// left down vert
	if( digit == '0' || digit == '2' || digit == '6' || digit == '8' )
	{
		r.topLeft.y = y + height;
		WinDrawRectangle( &r, kNumberCornerDiam );
	}

	width = kOneNumberWidth - kLinienStaerke*2;
	height = kOneNumberHeight / 10;

	r.topLeft.x = x + kLinienStaerke;
	r.extent.x = width;
	r.extent.y = kLinienStaerke;

	// up horz
	if( digit == '0' || digit == '2' || digit == '3' || digit == '5' || digit == '6' || digit == '7' || digit == '8' || digit == '9' )
	{
		r.topLeft.y = y;
		WinDrawRectangle( &r, kNumberCornerDiam );
	}
	// mid horz
	if( digit == '2' || digit == '3' || digit == '4' || digit == '5' || digit == '6' || digit == '8' || digit == '9' )
	{
		r.topLeft.y = y + (mid - kLinienStaerke/2);
		WinDrawRectangle( &r, kNumberCornerDiam );
	}

	// down horz
	if( digit == '0' || digit == '2' || digit == '3' || digit == '5' || digit == '6' || digit == '8' || digit == '9' )
	{
		r.topLeft.y = y + (kOneNumberHeight - kLinienStaerke);
		WinDrawRectangle( &r, kNumberCornerDiam );
	}
}

/*
 * FUNCTION:			MainFormDrawDays
 * DESCRIPTION:		computes and draws the remaining days
 * CALLED:				after the form is being drawn
 * 								when a new date was set
 * PARAMETERS:		frmP - pointer to the active form
 * RETURNS:				nothing
 */
static void MainFormDrawDays( FormType * frmP )
{
	RectangleType r;
	Char					szDate[5];		// max 9999 + null terminator
	UInt32				dueDays;
	UInt32				nowDays;
	DateType			now;
	UInt16				len;
	UInt16				x_offset;
	UInt16				i;

	r.topLeft.y = kNumberYOffset;
	r.extent.y = kOneNumberWidth+30;
	r.topLeft.x = 0;
	r.extent.x = 160;
	WinEraseRectangle( &r, 0 );

	if( DateToInt( gDueDate ) != noDate )
	{
		DateSecondsToDate( TimGetSeconds(), &now );
		dueDays = DateToDays( gDueDate );
		nowDays = DateToDays( now );
		if( nowDays <= dueDays )
		{
			dueDays -= nowDays;
			StrIToA( szDate, dueDays );
			len = StrLen( szDate );
			x_offset = (160 - (len*kOneNumberWidth + (len-1)*kNumberSpacing))/2;

			for( i = 0; i < len; i++ )
			{
				DrawDigit( szDate[i], x_offset, kNumberYOffset );
				x_offset += (kOneNumberWidth + kNumberSpacing);
			}
		}
	}
}

/*
 * FUNCTION:			MainFormInit
 * DESCRIPTION:		initializes the main form
 * CALLED:				when the main form is being opened (frmOpenEvent)
 * PARAMTERS:			frmP - pointer to the main form
 * RETURNS:				nothing
 */
static void MainFormInit( FormType * frmP )
{
	ControlType *	ctlP;
	Char *				labelP;

	// date trigger
	ctlP = FrmGetObjectPtr( frmP, FrmGetObjectIndex( frmP, MainDateTrigger ) );
	labelP = (Char *)CtlGetLabel( ctlP );
	if( DateToInt( gDueDate ) == noDate )
		StrCopy( labelP, NoDateString );
	else
		DateToDOWDMFormat( gDueDate.month, gDueDate.day, gDueDate.year + firstYear, gPreferedDateFormat, labelP );
	CtlSetLabel( ctlP, labelP );

	// alarm trigger
	ctlP = FrmGetObjectPtr( frmP, FrmGetObjectIndex( frmP, MainAlarmTrigger ) );
	labelP = (Char *)CtlGetLabel( ctlP );
	if( TimeToInt( gAlarmTime ) == noTime )
		StrCopy( labelP, NoAlarmString );
	else
		TimeToAscii( gAlarmTime.hours, gAlarmTime.minutes, gPreferedTimeFormat, labelP );
	CtlSetLabel( ctlP, labelP );
}

/*
 * FUNCTION:			MainFormSelectAlarm
 * DESCRIPTION:		lets the user select a time and sets an alarm
 * 								this routine also updates the passed control
 * CALLED:				when the user taps the alarm selector trigger
 * PARAMETERS:		ctlP - pointer to the alarm time trigger
 * RETURNS:				nothing
 */
static void MainFormSelectAlarm( ControlType * ctlP )
{
	DateTimeType 	dtbuf;
	Char *				labelP;
	UInt32				romVersion;
	UInt32				alarm;
	UInt16				cardno;
	UInt16				hour;
	UInt16				minute;
	LocalID				dbID;
	DmSearchStateType searchstate;
	TimeType			time;
	TimeType			unusedTime;
	Boolean				ok;

	if( DateToInt( gDueDate ) == noDate )
	{
		FrmAlert( SelectDueDateAlert );
		return;
	}

	if( TimeToInt( gAlarmTime ) == noTime )
	{
		TimSecondsToDateTime( TimGetSeconds(), &dtbuf );
		hour = dtbuf.hour;
		minute = dtbuf.minute;
	}
	else
	{
		hour = gAlarmTime.hours;
		minute = gAlarmTime.minutes;
	}

	FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVersion);
	if( romVersion < sysMakeROMVersion(3,1,0,sysROMStageRelease,0) )
	{
		time.hours = hour;
		time.minutes = minute;
		unusedTime = time;
		ok = SelectTimeV33( &time, &unusedTime, false, SelectAlarmTitle, 8 );
		if( ok )
		{
			hour = time.hours;
			minute = time.minutes;
		}
	}
	else
		ok = SelectOneTime( &hour, &minute, SelectAlarmTitle );

	if( ok )
	{
		gAlarmTime.hours = hour;
		gAlarmTime.minutes = minute;
		
		// set an alarm. if there is already one set then it will be replaced by this one

		dtbuf.second = 0;
		dtbuf.hour = hour;
		dtbuf.minute = minute;
		dtbuf.day = gDueDate.day;
		dtbuf.month = gDueDate.month;
		dtbuf.year = gDueDate.year + firstYear;

		alarm = TimDateTimeToSeconds( &dtbuf );

		if( errNone == DmGetNextDatabaseByTypeCreator( true, &searchstate, sysFileTApplication, appFileCreator, true, &cardno, &dbID ) )
		{
			if( !AlmSetAlarm( cardno, dbID, 0, alarm, true ) )
			{
				labelP = (Char *)CtlGetLabel( ctlP );
				TimeToAscii( hour, minute, gPreferedTimeFormat, labelP );
				CtlSetLabel( ctlP, labelP );
			}
		}
	}
}

/*
 * FUNCTION:			MainFormSelectDueDate
 * DESCRRIPTION:	pops up a dialog to let the user select a date
 * 								draws the passed control and updates the remaining days
 * CALLED:				when the user taps the date selector trigger
 * PARAMETERS:		ctlP - pointer to the date selector trigger
 * RETURNS:				nothing
 */
static void MainFormSelectDueDate( ControlType * ctlP )
{
	Char *		labelP;
	DateType	tmpDate;
	Int16			month;
	Int16			day;
	Int16			year;

	if( DateToInt( gDueDate ) == noDate )
	{
		DateSecondsToDate( TimGetSeconds(), &tmpDate );
		month = tmpDate.month;
		day = tmpDate.day;
		year = tmpDate.year + firstYear;
	}
	else
	{
		month = gDueDate.month;
		day = gDueDate.day;
		year = gDueDate.year + firstYear;
	}

	if( SelectDay( selectDayByDay, &month, &day, &year, SelectDueDateTitle ) )
	{
		gDueDate.month = month;
		gDueDate.day = day;
		gDueDate.year = year - firstYear;

		labelP = (Char *)CtlGetLabel( ctlP );
		DateToDOWDMFormat( month, day, year, gPreferedDateFormat, labelP );
		CtlSetLabel( ctlP, labelP );

		MainFormDrawDays( FrmGetActiveForm() );
	}
}

/*
 * FUNCTION:			MainFormClearAlarm
 * DESCRIPTION:		this routine delete the alarm if any
 * 								and update the alarm selector trigger
 * PARAMETERS:		frmP - pointer to the main form
 * RETURNS:				nothing
 */
static void	MainFormClearAlarm( FormType * frmP )
{
	DmSearchStateType searchstate;
	ControlType * 		ctlP;
	Char *						labelP;
	UInt16						cardno;
	LocalID						dbID;

	*((UInt16 *)&gAlarmTime) = noTime;
	//TimeToInt( gAlarmTime ) = noTime;
	if( errNone == DmGetNextDatabaseByTypeCreator( true, &searchstate, sysFileTApplication, appFileCreator, true, &cardno, &dbID ) )
	{
		if( !AlmSetAlarm( cardno, dbID, 0, 0, true ) )
		{
			ctlP = FrmGetObjectPtr( frmP, FrmGetObjectIndex( frmP, MainAlarmTrigger ) );
			labelP = (Char *)CtlGetLabel( ctlP );
			StrCopy( labelP, NoAlarmString );
			CtlSetLabel( ctlP, labelP );
		}
	}
}

/*
 * FUNCTION:			MainFormHandleEvent
 * DESCRIPTION:		this is the event handler for the main form
 * CALLED:				each time an unhandled event arrives
 * PARAMETERS:		eventP - pointer to the event
 * RETURNS:				true if the event was completely handled
 * 								false if not.
 */
static Boolean MainFormHandleEvent( EventType * eventP )
{
  Boolean 		handled = false;
  FormType	* frmP;

	switch( eventP->eType ) 
	{
		case menuEvent:
			if( eventP->data.menu.itemID == OptionsAbout )
			{
				MenuEraseStatus (0);
				frmP = FrmInitForm( AboutDialog );
				FrmDoDialog( frmP );
				FrmDeleteForm( frmP );
				handled = true;
			}
			else if( eventP->data.menu.itemID == OptionsClearAlarm )
			{
				MenuEraseStatus(0);
				frmP = FrmGetActiveForm();
				MainFormClearAlarm( frmP );
				handled = true;
			}
			break;

		case frmOpenEvent:
			frmP = FrmGetActiveForm();
			MainFormInit( frmP );
			FrmDrawForm ( frmP );
			MainFormDrawDays( frmP );
			handled = true;
			break;

		case ctlSelectEvent:
			if( eventP->data.ctlSelect.controlID == MainDateTrigger )
				MainFormSelectDueDate( eventP->data.ctlSelect.pControl );
			else
				MainFormSelectAlarm( eventP->data.ctlSelect.pControl );
			handled = true;
			break;

		default:
			break;
		
	}
	
	return handled;
}


/*
 * FUNCTION:			RomVersionCompatible
 * DESCKRPTION:		This routine checks that a ROM version is 
 * 								meet our minimum requirement
 * 								it also checks the screen mode and displays 
 * CALLED:				when our app is being launched
 * PARAMETERS:		requiredVersion ... should be 3.5
 * 								launchFlags ... should be sysAppLaunchCmdNormal
 * RETURNS:				0 - if the system version is 3.1 or higher
 * 								else sysErrRomIncompatible
 */
static UInt16 RomVersionCompatible( UInt32 requiredVersion, UInt16 launchFlags )
{
	UInt32 actualValue;

	// See if we're on in minimum required version of the ROM or later.
	FtrGet(sysFtrCreator, sysFtrNumROMVersion, &actualValue );
	if (actualValue < requiredVersion)
	{
		if ((launchFlags & (sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp)) ==
			(sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp))
		{
			FrmAlert (RomIncompatibleAlert);
		
			// Pilot 1.0 will continuously relaunch this app unless we switch to 
			// another safe one.
			if (actualValue < sysMakeROMVersion(2,0,0,sysROMStageRelease,0))
				AppLaunchWithCommand(sysFileCDefaultApp, sysAppLaunchCmdNormalLaunch, NULL);
		}
		
		return (sysErrRomIncompatible);
	}

	return (0);
}

/*
 * FUNCTION:			AppHandleEvent
 * DESCRIPTION:		this is the application event handler
 * 								it corresponds only to frmLoadEvent
 * RETURNS:				true if the event was handled, else false
 */
static Boolean AppHandleEvent( EventType * eventP )
{
	UInt16 			formId;
	FormType * 	frmP;


	if( eventP->eType == frmLoadEvent )
	{
		// Load the form resource.
		formId = eventP->data.frmLoad.formID;
		frmP = FrmInitForm(formId);
		FrmSetActiveForm(frmP);

		// Set the event handler for the form.  The handler of the currently
		// active form is called by FrmHandleEvent each time is receives an
		// event.
		switch (formId)
		{
			case MainForm:
				FrmSetEventHandler(frmP, MainFormHandleEvent);
				break;

			default:
				break;

		}
		return true;
	}
	
	return false;
}

/*
 * FUNCTION:			AppEventLoop
 * DESCRIPTION:		well ... this is the heart of our event driven app
 */
static void AppEventLoop(void)
{
	UInt16		error;
	EventType event;

	do
	{
		EvtGetEvent(&event, evtWaitForever);
		
		if (! SysHandleEvent(&event))
			if (! MenuHandleEvent(0, &event, &error))
				if (! AppHandleEvent(&event))
					FrmDispatchEvent(&event);

	}
	while (event.eType != appStopEvent);
}

/*
 * FUNCTION:			AppStart
 * DESCRIPTION:		Initializes the radom number generator
 * CALLED:				when our app is launched and to be run
 * RETURNS:				0 - at the moment 
 */
static UInt16 AppStart( void )
{
	CountDownPrefsType prefs;
	UInt16	prefsSize;
#ifdef CONFIG_SONY
	Err			error = 0;
	SonySysFtrSysInfoP sysP;
#endif /* CONFIG_SONY */

	// Read the saved preferences / saved-state information.
	prefsSize = sizeof(CountDownPrefsType);
	if( PrefGetAppPreferences( appFileCreator, appPrefID, &prefs, &prefsSize, false) != noPreferenceFound)
	{
		gDueDate = prefs.due;
		gAlarmTime = prefs.alarm;
	}
	else
	{
		DateToInt( gDueDate ) = noDate;
		TimeToInt( gAlarmTime ) = noTime;
	}

	gPreferedDateFormat = (DateFormatType)PrefGetPreference( prefDateFormat );
	gPreferedTimeFormat = (TimeFormatType)PrefGetPreference( prefTimeFormat );

#ifdef CONFIG_SONY
	gSonyLibRefNum = 0;
	if( !FtrGet( sonySysFtrCreator, sonySysFtrNumSysInfoP, (UInt32 *)&sysP ) )
		if( sysP->libr & sonySysFtrSysInfoLibrHR )
			if( (error = SysLibFind( sonySysLibNameHR, &gSonyLibRefNum )) )
				if( error == sysErrLibNotFound )
					error = SysLibLoad( 'libr', sonySysFileCHRLib, &gSonyLibRefNum );

	if( gSonyLibRefNum )
	{
		UInt32 w, h, d;
		
		HROpen( gSonyLibRefNum );
		w = hrWidth;
		h = hrHeight;
		d = 8;
		HRWinScreenMode( gSonyLibRefNum, winScreenModeSet, &w, &h, &d, NULL );
	}
#endif /* CONFIG_SONY */

	return 0;
}

/*
 * FUNCTION:			AppStop
 * DESCRIPTION:		closes all opened forms
 */
static void AppStop( void )
{
  CountDownPrefsType prefs;

	prefs.due = gDueDate;
	prefs.alarm = gAlarmTime;
	PrefSetAppPreferences (appFileCreator, appPrefID, appPrefVersionNum, &prefs, sizeof (prefs), false );

#ifdef CONFIG_SONY
	if( gSonyLibRefNum )
	{
		HRWinScreenMode( gSonyLibRefNum, winScreenModeSetToDefaults, NULL, NULL, NULL, NULL );
		HRClose( gSonyLibRefNum );
	}
#endif
}

/*
 * FUNCTION:			DisplayAlarm
 * DESCRIPTION:		this function shows a simple dialog box
 * CALLED:				when we was launched with sysAppLaunchCmdDisplayAlarm
 * PARAMETERS:		see palm SDK
 * RETURNS:				nothing
 */
static void	DisplayAlarm( SysDisplayAlarmParamType *cmdPBP )
{
	FormType * frmP = FrmInitForm( AlarmDialog );
	FrmDoDialog( frmP );
	FrmDeleteForm( frmP );
}

/*
 * FUNCTION:			PilotMain
 * DESCRIPTION:		this is the entry point for our ap
 * ...
 */
UInt32 PilotMain( UInt16 cmd, void * cmdPBP, UInt16 launchFlags )
{
	UInt16 error;

	if( cmd == sysAppLaunchCmdNormalLaunch )
	{
		error = RomVersionCompatible (ourMinVersion, launchFlags);
		if( error ) return (error);

		error = AppStart();
		if (error) return error;
				
		FrmGotoForm(MainForm);
		AppEventLoop();
		AppStop();
	}
	else if( cmd == sysAppLaunchCmdAlarmTriggered )
	{
		// nothing to do yet
		// maybe we should play a sound
	}
	else if( cmd == sysAppLaunchCmdDisplayAlarm )
	{
		DisplayAlarm( (SysDisplayAlarmParamType *)cmdPBP );
	}
	
	return 0;
}
