/*
 *   $Id: history.c,v 1.5 2006/09/12 19:40:56 tvoverbe Exp $
 * 
 *   Scientific Calculator for Palms.
 *   Copyright (C) 2000 Ondrej Palkovsky
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *   Boston, MA  02110-1301, USA.
 * 
 *  You can contact me at 'ondrap@penguin.cz'.
*/

#include <PalmOS.h>

#include <string.h>

#include "calcDB.h"
#include "stack.h"
#include "defuns.h"
#include "history.h"
#include "display.h"

static DmOpenRef dbRef;

UInt16
history_total(void) MLIB;
UInt16
history_total(void)
{
	return DmNumRecords(dbRef);
}

void
history_add_line(char *line)
{
	tHistory *item,*newitem;
	UInt16 size;
	MemHandle newHandle;
	UInt16 recordNumber;
	
	size = sizeof(*item) + StrLen(line) + 1;
	item = MemPtrNew(size);
	item->isrpn = false;
	StrCopy(item->u.text,line);
	
	recordNumber = 0;
	newHandle = DmNewRecord(dbRef,&recordNumber,size);	
	newitem = MemHandleLock(newHandle);
	DmWrite(newitem,0,item,size);
	MemHandleUnlock(newHandle);
	DmReleaseRecord(dbRef,recordNumber,true);
	
	MemPtrFree(item);
}

/***********************************************************************
 *
 * FUNCTION:     history_add
 * 
 * DESCRIPTION:  Add a record in a records database
 *
 * PARAMETERS:   item - the record that should be added
 *
 * RETURN:       None
 *      
 ***********************************************************************/
void
history_add_item(Trpn item)
{
	dbStackItem *packitem;
	UInt16 size,psize;
	MemHandle newHandle;
	UInt16 recordNumber;
	tHistory *histitem,*newitem;
	
	packitem = rpn_pack_record(item);
	if (!packitem)
	    return;
	size = sizeof(*histitem) + packitem->datasize;
	psize = sizeof(*packitem) + packitem->datasize;
	histitem = MemPtrNew(size);
	if (!histitem) {
	    MemPtrFree(packitem);
	    return;
	}
	histitem->isrpn = true;
	memcpy((void *)&histitem->u.item,(void *)packitem,psize);
	
	recordNumber = 0;
	newHandle = DmNewRecord(dbRef,&recordNumber,size);
	if (!newHandle) {
	    MemPtrFree(packitem);
	    MemPtrFree(histitem);
	    return;
	}
	newitem = MemHandleLock(newHandle);
	DmWrite(newitem,0,histitem,size);
	MemHandleUnlock(newHandle);
	DmReleaseRecord(dbRef,recordNumber,true);
	
	MemPtrFree(packitem);
	MemPtrFree(histitem);
}

/***********************************************************************
 *
 * FUNCTION:     history_shrink
 * 
 * DESCRIPTION:  Shrink the history detabase to a defined
 *               number of records
 *
 * PARAMETERS:   How many records leave in database
 *
 * RETURN:       None
 *      
 ***********************************************************************/
void 
history_shrink(Int16 count)
{
	Int16 i;

	if (DmNumRecords(dbRef)==0)
	  return;
	i = DmNumRecords(dbRef)-1;
	while (i>=count) {
		DmRemoveRecord(dbRef,i--);
	}
}


/***********************************************************************
 *
 * FUNCTION:     history_get
 * 
 * DESCRIPTION:  Return a history record
 *
 * PARAMETERS:   num - how old the item is
 *
 * RETURN:       item
 *      
 ***********************************************************************/
Trpn 
history_get_item(UInt16 num)
{
	MemHandle recordHandle;
	tHistory *historyitem;
	Trpn result;
	
	recordHandle = DmQueryRecord(dbRef,num);
	historyitem = MemHandleLock(recordHandle);
	result = rpn_unpack_record(&historyitem->u.item);
	MemHandleUnlock(recordHandle);
	
	return result;
}

char *
history_get_line(UInt16 num)
{
	MemHandle recordHandle;
	tHistory *historyitem;
	char *result;
	
	recordHandle = DmQueryRecord(dbRef,num);
	historyitem = MemHandleLock(recordHandle);
	result = MemPtrNew(StrLen(historyitem->u.text) + 1);
	StrCopy(result,historyitem->u.text);
	MemHandleUnlock(recordHandle);
	
	return result;
}

Boolean
history_isrpn(UInt16 num)
{
	MemHandle recordHandle;
	tHistory *historyitem;
	Boolean result;
	
	recordHandle = DmQueryRecord(dbRef,num);
	historyitem = MemHandleLock(recordHandle);
	result = historyitem->isrpn;
	MemHandleUnlock(recordHandle);
	
	return result;
}


/***********************************************************************
 *
 * FUNCTION:     history_command
 * 
 * DESCRIPTION:  This function is evaluated directly from a normal
 *               mathematical equation
 *
 * PARAMETERS:   On stack - 1 number convertible to integer
 *
 * RETURN:       On stack - value from history database
 *      
 ***********************************************************************/
CError
history_command(Functype *func,CodeStack *stack)
{
	CError err;
	UInt32 arg;
	
	if ((err=stack_get_val(stack,&arg,integer)))
	  return err;	
	if (arg > DmNumRecords(dbRef))
	  return c_badarg;

	if (!history_isrpn(arg))
		return c_badarg;
	stack_push(stack,history_get_item(arg));
	
	return c_noerror;
}

static Err history_open_byid(LocalID dbid) MLIB;
static Err
history_open_byid(LocalID dbid)
{
        dbRef=DmOpenDatabase(CARDNO,dbid,dmModeReadWrite);
        if (dbRef)
          return DmGetLastErr();                        
        return 0;
}

/***********************************************************************
 *
 * FUNCTION:     history_open
 * 
 * DESCRIPTION:  Open a history database
 *
 * PARAMETERS:   None
 *
 * RETURN:       0 on success
 *      
 ***********************************************************************/
Int16 
history_open(void)
{
	Err err;
	LocalID dbid;
	UInt16 version;
	UInt32 creator;
	UInt32 type;
	
	dbid=DmFindDatabase(CARDNO,HISTORYDBNAME);
	if (dbid) {  /* Database exists */
		DmDatabaseInfo(CARDNO,dbid,
			       NULL, /* name */
                               NULL, /* attrib */
                               &version, /* version */
                               NULL, /* crDate */
                               NULL, /* modDate */
                               NULL, /* bckUpDate */
                               NULL, /* modNum */
                               NULL, /* appinfoID */
                               NULL, /* sortInfoID */
                               &type, /* Type */
                               &creator); /* Creator */

		if (version==HIST_DB_VERSION && creator==LIB_ID && type==DBTYPE)
		  return history_open_byid(dbid);
		
		/* Database exists, but with uncorrect version */
		err=DmDeleteDatabase(CARDNO,dbid);
		if (err)
		  return err;
	}
	
	/* Database doesn't exist or old version */
	err=DmCreateDatabase(CARDNO,HISTORYDBNAME,LIB_ID,DBTYPE,false);
	if (err)
	  return err;
	
	dbid=DmFindDatabase(CARDNO,HISTORYDBNAME);
	
	version=HIST_DB_VERSION;
	DmSetDatabaseInfo(CARDNO,dbid,
			  NULL, /* name */
			  NULL, /* attrib */
			  &version, /* version */
			  NULL, /* crDate */
			  NULL, /* modDate */
			  NULL, /* bckUpDate */
			  NULL, /* modNum */
			  NULL, /* appinfoID */
			  NULL, /* sortInfoID */
			  NULL, /* Type */
			  NULL); /* Creator */
	return history_open_byid(dbid);
}

/***********************************************************************
 *
 * FUNCTION:     history_close
 *
 * DESCRIPTION:  Close a history database
 *
 * PARAMETERS:   None
 *
 * RETURN:       0 on success
 *      
 ***********************************************************************/
Int16 
history_close(void)
{
	history_shrink(HISTORY_RECORDS);
	return DmCloseDatabase(dbRef);
}
