/*  OpenChess.c -- OpenChess PalmOS  display interface
 *
 *  OpenChess version 2.0, is a CHESS game for Palm Pilot, adapted
 *  from GNU Chess 2, for Unix/DOS environments
 *  Copyright (C) 1986, 1987, 1988 Free Software Foundation, Inc.
 *  Copyright (C) 2002, Son Altesse.
 *  Copyright (C) 2005, Olaf Richter.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 
#include <PalmOS.h>
#include <Bitmap.h>
#include <StringMgr.h>
#include <Libraries/PalmOSGlue/PalmOSGlue.h>

#include "SonySDK/SonyHRLib.h"
#include "SonySDK/SonySystemFtr.h"

#include "OpenChess_res.h"
#include "OpenChess.h"
#include "gnuchess2.h"

#define SCR_TITLE       13
#define DENSITY(x)      (x*prgg.density)

#define fullSq 			DENSITY(20)
#define normSq 			DENSITY(18)
#define fullOffset		0
#define normOffset 		DENSITY(4)
#define wdCoordinateX	DENSITY(3)
#define wdFigureY		DENSITY(5)
#define wdLetterY		DENSITY(3)
#define wdLine			DENSITY(1)
#define wdPieces		DENSITY(15)
#define offSqWhite		DENSITY(90)
#define PieceHeight		DENSITY(15)
#define TitleHeight		DENSITY(SCR_TITLE)


#define X_TEXT          1	    
#define EX_TEXT         148
#define Y_TEXT          149   
#define EY_TEXT         (160-Y_TEXT+1)

#define NoSqSelected    -1

// Program globals
short	PromotionSelection = 0;

#define SECTION_TWO  __attribute__ ((section ("section2")))
// Program prototypes
static Boolean ApplicationHandleEvent(EventPtr event) SECTION_TWO;
Boolean frmMain_HandleEvent(EventPtr event) SECTION_TWO;
Boolean frmAbout_HandleEvent(EventPtr event) SECTION_TWO;
Boolean frmMessage_HandleEvent(EventPtr event) SECTION_TWO;
Boolean frmEditBrd_HandleEvent(EventPtr event);
Boolean frmOptions_HandleEvent(EventPtr event) SECTION_TWO;
Boolean frmPrefs_HandleEvent(EventPtr event) SECTION_TWO;
Boolean frmPieces_HandleEvent(EventPtr event);
Boolean frmPiecesTaken_HandleEvent(EventPtr event);
Boolean frmDatabase_HandleEvent(EventPtr event) SECTION_TWO;
void ManualMove(void);
Boolean CheckBoardEdit(void);
void SetEndGame(short side, short ply);
TOpenChessPrefs *GetPrefsPtr(void);
TGame *GetGamePtr(void);
TGlobals *GetGameGlobals(void);
void CheckRules(short sq);
Char *MoveString(short gcnt, unsigned short flags);
void AllocHashMem(void);
void FreeHashMem(void);

// Ext. segments
void *GetCtrlPtr(const FormType *formP, UInt16 objIndex) SECTION_TWO;
void SetDBBackupBit( DmOpenRef dbP ) SECTION_TWO;
Err OpenChessGetDatabase ( DmOpenRef *dbPP, UInt16 mode ) SECTION_TWO;
void ReadGameFromDB(UInt16 index) SECTION_TWO;
UInt16 SaveGameToDB(UInt16 index) SECTION_TWO;
void GetOpenings(void) SECTION_TWO;
void ShowMessage(char *s) SECTION_TWO;
void BackToMain(void) SECTION_TWO;
void LoadPrefs(TOpenChessPrefs *pp) SECTION_TWO;
void SavePrefs(TOpenChessPrefs *pp) SECTION_TWO;
void DrawDbListItem(Int16 iItem, RectangleType *bounds, Char **itemsText) SECTION_TWO;
void UnselectPieces(void) SECTION_TWO;
void GiveHint(void) SECTION_TWO;
void SetFullMode(Boolean fm) SECTION_TWO;
void FreeLibMem(void) SECTION_TWO;
void FreeBookMem(void) SECTION_TWO;
void GetProgStrings(void) SECTION_TWO;
Boolean getsq(Int16 ScreenX, Int16 ScreenY, UInt16 *sq) SECTION_TWO;
void Copy2screen(Coord xs, Coord ys, Coord ex, Coord ey,
                 Coord xd, Coord yd, WinDrawOperation mode) SECTION_TWO;
char *SkipComments(Int16 i, char *txtPtr) SECTION_TWO;
void EndOfOpening(Boolean ok) SECTION_TWO;
void Back(void) SECTION_TWO;
void DisplayStatus(void) SECTION_TWO;
void PGNExport(void) SECTION_TWO;
void OpenUI(void) SECTION_TWO;
void SetLibText(void) SECTION_TWO;
Boolean DoEvents(void) SECTION_TWO;
void DrawBmp(UInt32 bmpID, Coord x, Coord y) SECTION_TWO;
void DrawButton(FormPtr form, UInt32 bmpID, UInt32 btnID) SECTION_TWO;

TOpenChessPrefs prefs;
TGame           hdr;
TGlobals	    prgg;

/****************************************************************************/
void * GetCtrlPtr(const FormType *formP, UInt16 objIndex) //S2
{
  return FrmGetObjectPtr(formP, FrmGetObjectIndex (formP, objIndex));
}

/****************************************************************************/
void SetDBBackupBit( DmOpenRef dbP ) // S2
{
LocalID     dbID;
UInt16      cardNo;
UInt16      attributes;
UInt16      versionP;

DmOpenDatabaseInfo( dbP, &dbID, NULL, NULL, &cardNo, NULL );
DmDatabaseInfo( cardNo, dbID, NULL, &attributes, &versionP, NULL,
                NULL, NULL, NULL, NULL, NULL, NULL, NULL);
versionP = VERSION_NUM;
attributes |= dmHdrAttrBackup;
DmSetDatabaseInfo( cardNo, dbID, NULL, &attributes, &versionP, NULL,
                  NULL, NULL, NULL, NULL, NULL, NULL, NULL);

}

/****************************************************************************/
Err OpenChessGetDatabase ( DmOpenRef *dbPP, UInt16 mode ) // S2
{
Err             error = false;
DmOpenRef       dbP;

*dbPP = NULL;
dbP = DmOpenDatabaseByTypeCreator (OpenChessDBType, sysFileCOpenChess, mode);
if (!dbP) {
    error = DmCreateDatabase ( 0, OpenChessDBName, sysFileCOpenChess, OpenChessDBType, false );
    if (error) return error;
    dbP = DmOpenDatabaseByTypeCreator( OpenChessDBType, sysFileCOpenChess, mode );
    if (!dbP) return ( true );
    SetDBBackupBit( dbP ); // Set the backup bit to enable hotsyncs
    }
*dbPP = dbP;
return false;
}

/****************************************************************************/
void GetGame(MemPtr GamePtr)   // not S2
{
short                   sq;
unsigned short          m;
short                   i;
MemPtr                  p;

  MemMove(&hdr, GamePtr, sizeof(hdr));
  for (sq = 0; sq < 64; sq++)
  {
     m = hdr.board[sq];
     board[sq] = (m >> 8);
     color[sq] = (m & 0xFF);
  }
  p = GamePtr + sizeof(hdr);
  for (i = 0; i <= hdr.GameCnt; i++, p += sizeof(struct GameRec))
      MemMove(&GameList[i], p, sizeof(struct GameRec));
  if (hdr.TimeControl.clock[white] > 0)
    hdr.TCflag = true;
  InitializeStats();
  Sdepth = 0;
}

/****************************************************************************/
void SaveGame(UInt16 index)   // not S2
{
short         sq, i;
MemHandle     RecHandle;
MemPtr        RecPtr;
UInt16        RecSz;
TGlobals      *pgp = GetGameGlobals();

  RecHandle = DmGetRecord(pgp->OpenChessDB, index);
  RecPtr = MemHandleLock(RecHandle);
  for (sq = 0; sq < 64; sq++) 
    hdr.board[sq] = 256 * board[sq] + color[sq];
  DmWrite(RecPtr, 0, &hdr, sizeof(hdr));
  RecSz = sizeof(hdr);
  for (i = 0; i <= hdr.GameCnt; i++, RecSz += sizeof(struct GameRec))
      DmWrite(RecPtr, RecSz, &GameList[i], sizeof(struct GameRec));
  MemHandleUnlock(RecHandle);
  DmReleaseRecord(pgp->OpenChessDB, index, true);
}

/****************************************************************************/
void ReadGameFromDB(UInt16 index) // S2
{
  MemHandle   RecHand;
  MemPtr      RecPtr;
  TGlobals    *pgp = GetGameGlobals();

  RecHand = DmQueryRecord(pgp->OpenChessDB, index);    // # of Game to load
  RecPtr = MemHandleLock(RecHand);
  if (RecPtr == 0) 
    FrmCustomAlert(AlertError, SkipComments(12, pgp->ProgStrings), "", "");
  else 
    GetGame(RecPtr);   // Process the record as in GNUchess's uxdsp.c GetGame
  MemHandleUnlock(RecHand);
}

/****************************************************************************/
UInt16 SaveGameToDB(UInt16 index) // S2
{
  TGame     *gp = GetGamePtr();
  TGlobals  *pgp = GetGameGlobals();

  if (DmQueryRecord(pgp->OpenChessDB, index))
    DmRemoveRecord(pgp->OpenChessDB, index);

  DmNewRecord(pgp->OpenChessDB, &index,
    sizeof(hdr) + (gp->GameCnt + 1) * sizeof(struct GameRec));
  DmReleaseRecord(pgp->OpenChessDB, index, true);
  SaveGame(index);
  return index;
}

/****************************************************************************/
void LoadPrefs(TOpenChessPrefs *pp) // S2
{
  UInt16  PrefsSz = 0;
  Int16   versionN;

  versionN = PrefGetAppPreferences(sysFileCOpenChess, 0, NULL, &PrefsSz, false);
  if ((versionN == VERSION_NUM) && (PrefsSz == sizeof(prefs)))
  {
    PrefGetAppPreferences(sysFileCOpenChess, 0, pp, &PrefsSz, false);
  }
  else
  {
    // default Language from OS settings 
    if(PrefGetPreference(prefVersion) >= preferenceDataVer9) 
    {
      UInt32 ui32;
      LmLocaleType Loc;
    
      ui32 = PrefGetPreference(prefLocale);
      *((UInt32 *)&Loc) = ui32;
      switch(Loc.language)
      {
	case lGerman:
          pp->Langu = resGerman;
          break;
	case lFrench:
          pp->Langu = resFrench;
          break;
	case lSpanish:
          pp->Langu = resSpanish;
          break;
    case lItalian:      
          pp->Langu = resItalian;
          break;
    case lDutch:      
          pp->Langu = resDutch;
          break;
	case lEnglish:
	default:
          pp->Langu = resEnglish;
          break;
      }
    }
	// delete V1.0 DB to prevent errors
	if(versionN <= VERSION_NUM_20)
       DmDeleteDatabase(0, DmFindDatabase(0, OpenChessDBName));

    pp->fullModeMsg =
    pp->SaveIndex = 1;
/*  pp->rgbL.index = 1;
    pp->rgbL.r     = 192;
    pp->rgbL.g     = 220;
    pp->rgbL.b     = 192;
    pp->rgbD.index = 5;
    pp->rgbD.r     = 224;
    pp->rgbD.g     = 128;
    pp->rgbD.b     = 64;*/
  }
}

/****************************************************************************/
void SavePrefs(TOpenChessPrefs *pp) // S2
{
  PrefSetAppPreferences(sysFileCOpenChess, 0, VERSION_NUM,
                        pp, sizeof(prefs), false);
}

/****************************************************************************/
void GetOpenings(void) // S2
{
  Int16            i,j,*p = NULL;
  struct BookEntry *b, *lastBook;
  UInt16           Cnt = 0,
                   Recs = 0,
                   **List = NULL;
  LocalID          dbID = 0;
  TOpenChessPrefs  *pp  = GetPrefsPtr();
  TGlobals         *pgp = GetGameGlobals();
  TGame            *gp  = GetGamePtr();
  char             *txtPtr;

  FreeBookMem();
  FreeHashMem();
  
  if((pp->OnlyTrainer && gp->TrainerMode) || pp->NormGames)
  {
    if((gp->opponent == white) && StrLen(pp->OLibWhite))
	  dbID = DmFindDatabase (0, pp->OLibWhite);
    else
      if((gp->opponent == black) && StrLen(pp->OLibBlack))
	    dbID = DmFindDatabase (0, pp->OLibBlack);

    if(dbID)
    {
      pgp->BookDB = DmOpenDatabase (0, dbID, dmModeReadWrite);
      if(pgp->BookDB)
        pgp->BookRecs = DmNumRecords(pgp->BookDB);
      if(pgp->BookRecs)
      {
        List =  
        pgp->BookDataList = (UInt16 **)MemGluePtrNew((UInt32)(pgp->BookRecs * sizeof(UInt16 *)));
	if(List)  
	{
          for(i = 0; i < pgp->BookRecs; i++)
          {
            *List = (UInt16 *)MemHandleLock(DmQueryRecord(pgp->BookDB, i));
            CRASHIF(!*List, "PDB Lock.");  
            pgp->OpenCnt += (UInt32)**List++;
          }   
          List = pgp->BookDataList;
          p = *List;
          pgp->Book = pgp->BookPtr = 
            (struct BookEntry *)MemGluePtrNew((UInt32)(pgp->OpenCnt * sizeof(struct BookEntry)));
	}
      }
      if(!pgp->Book)
      {
        Char msg[200];
          
        StrPrintF(msg, SkipComments(29, pgp->ProgStrings), 
          (gp->opponent == white ? pp->OLibWhite : pp->OLibBlack));  
        FrmCustomAlert(AlertError+(prefs.Langu*resLF), msg, "", "");
        pgp->BookRecs = 0;
        if(pgp->BookDataList)
        {
          MemPtrFree(pgp->BookDataList);
          List = pgp->BookDataList = NULL;
        }
        if(pgp->BookDB)
        {
          DmCloseDatabase(pgp->BookDB);
          pgp->BookDB = 0;
        }
        pgp->BookDB = NULL;
      }
    }
  }
    
  if(!pgp->Book)
  {
    pgp->BookHdl = DmGetResource('Data', OpenBookRes);
    pgp->BookData = (UInt16 *)MemHandleLock(pgp->BookHdl);
    pgp->OpenCnt = (UInt32)*pgp->BookData;
    p = pgp->BookData;
    pgp->Book = pgp->BookPtr = 
      (struct BookEntry *)MemPtrNew((UInt32)(pgp->OpenCnt * sizeof(struct BookEntry)));
    CRASHIF(!pgp->Book, "No memory.");  
  }

  // use first move as hint
  Cnt = *p++;
  if(gp->GameCnt < 0)
    gp->hint = *p; 

  // Set up the book structures
  b = pgp->Book;
  
  while(Cnt)
  {  
    lastBook = b;
    for(i = 0; i < Cnt; i++)
    {
      b->comments = NULL;
      b->mv = p;
      b++;
      while(*(p++) != 0);        // Seek next opening
    }
    // Comments
    if(List)
    {
      txtPtr = (Char *)p;
      b = lastBook;
      for(i = 0; i < Cnt; i++)
      {
	    b->comments = txtPtr;
	    if((*txtPtr != '\0') && (pgp->LibRootText == NULL))
	      pgp->LibRootText = txtPtr;
	    j = 0;
	    p = b->mv;
	    while(*p++)
	      j++;
	    txtPtr = SkipComments(j+1, txtPtr);
	    b++;
      }
    }
    if(++Recs < pgp->BookRecs)
    {
      List++;
      p = *List;
      Cnt = *p++;
    }
    else
    {
      break;
    }
  }
}

/****************************************************************************/
static int StartApplication(void) // not S2
{
  Err         err;
  UInt32      romVersion;

// Init vars
  MemSet(&prgg, sizeof(prgg), 0);
  MemSet(&hdr, sizeof(hdr), 0);
  MemSet(&prefs, sizeof(prefs), 0);
  
// General GnuChess init  
  OpenChessMain();    
  
// Read preferences    
  LoadPrefs(&prefs);  

// check OS  
  FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVersion);
  if (romVersion < 0x03000000) 
  {
	if(FrmAlert(AlertOldOS+(prefs.Langu*resLF)))
	  return 1;
  }
  
// read strings (language dep.)
  GetProgStrings();
  
  if((history == NULL) || (prgg.ProgStrings == NULL))
  {
    FrmAlert(AlertErrorNoMemory);
    return 1;
  }
  
  err = OpenChessGetDatabase(&prgg.OpenChessDB, dmModeReadWrite);    // Open the OpenChessDB database
  if(err)
    return err;

  if (DmNumRecords(prgg.OpenChessDB))
  {
    ReadGameFromDB(prefs.DbGameIndex);   // If not first time, read last game
  }
  else
  {
    hdr.sqSel.start = hdr.sqSel.end = NoSqSelected;
    prefs.DbGameIndex = SaveGameToDB(0);
  }

  OpenUI();

  GetOpenings();

  SetFullMode(prefs.fullMode);

  if (prefs.editMode)
    FrmGotoForm(frmEditBrd);
  else
    BackToMain();
  return 0;
}

/****************************************************************************/
void ShowMessage(char *s) // S2
{
  RectangleType   rect;
  TOpenChessPrefs *pp = GetPrefsPtr();
  TGame     *gp = GetGamePtr();
  TGlobals  *pgp = GetGameGlobals();

  if (pp->fullMode)
  {
    if(gp->endgame)
    {
      FrmCustomAlert(AlertGameEnd+(pp->Langu*resLF), 
        SkipComments(gp->endgame, pgp->ProgStrings), "", "");
      FrmGotoForm(frmMain);
    }
  }
  else
  {
    if(pgp->HRrefNum)
    {
      rect.topLeft.x = (X_TEXT*pgp->density)-1;
      rect.topLeft.y = (Y_TEXT*pgp->density)-1;
      rect.extent.x = (EX_TEXT*pgp->density)+2;
      rect.extent.y = (EY_TEXT*pgp->density)+2;
      HRFntSetFont(pgp->HRrefNum, hrStdFont);
      HRWinEraseRectangle (pgp->HRrefNum, &rect, 0); 
      HRWinDrawTruncChars(pgp->HRrefNum, 
	    s, StrLen(s), rect.topLeft.x, rect.topLeft.y, rect.extent.x);
    }
    else
    {
      rect.topLeft.x = X_TEXT;
      rect.topLeft.y = Y_TEXT;
      rect.extent.x = EX_TEXT;
      rect.extent.y = EY_TEXT;
      FntSetFont(stdFont);
      WinEraseRectangle (&rect, 0); 
      WinGlueDrawTruncChars(s, StrLen(s), rect.topLeft.x, rect.topLeft.y, rect.extent.x);
    }
  }  
}

/****************************************************************************/
void Copy2screen(Coord xs, Coord ys, Coord ex, Coord ey,
                 Coord xd, Coord yd, WinDrawOperation mode) // S2
{
  RectangleType   rect;
  WinHandle       oldWh = NULL;
  TGlobals        *pgp = GetGameGlobals();

  rect.topLeft.x = xs;
  rect.topLeft.y = ys;
  rect.extent.x = ex;
  rect.extent.y = ey;
  if(pgp->isHiRes) 
  {
    oldWh = WinSetDrawWindow(pgp->DispWindowH);
    WinSetCoordinateSystem(kCoordinatesNative);
  }
  if(pgp->HRrefNum)
	HRWinCopyRectangle(pgp->HRrefNum, pgp->OffscreenWinH, pgp->DispWindowH, 
	                   &rect, xd, yd, mode);
  else	  
    WinCopyRectangle(pgp->OffscreenWinH, pgp->DispWindowH, &rect, xd, yd, mode);
  if(pgp->isHiRes) 
  {
    WinSetDrawWindow(oldWh);
    WinSetCoordinateSystem(kCoordinatesStandard);
  }
}

/****************************************************************************/
void DrawPiece(Int16 sq)
{
  short r,c, offset;
  Coord x, y;
  RectangleType   rect;
  WinHandle oldWh = NULL;

  if(sq < 0) 
    return;

  if(hdr.reverse)
    r = 7-row[sq];
  else
    r = row[sq];
  if (hdr.reverse)
    c = 7-column[sq];
  else
    c = column[sq];

  r = 7-r;

// Draw the board square
  offset = (((c + r) % 2) ? fullSq : 0) + (fullSq - prgg.szSq) / 2;      
  x = c * prgg.szSq;
  y = r * prgg.szSq;
  x += prgg.offBoard;
  y += prgg.offBoard;
  Copy2screen(offSqWhite + (fullSq - prgg.szSq) / 2, offset, prgg.szSq, prgg.szSq, x, y, winPaint);

  if ((hdr.sqSel.start == sq) || (hdr.sqSel.end == sq))
  {
    rect.topLeft.x = x + wdLine;
    rect.topLeft.y = y + wdLine;
    rect.extent.x =
    rect.extent.y = prgg.szSq - (2 * wdLine);
    if (hdr.sqSel.start == sq)
    {
// Display the start selection mark
      if(prgg.isHiRes)
      {
        oldWh = WinSetDrawWindow(prgg.DispWindowH);
        WinSetCoordinateSystem(kCoordinatesNative);
      }
      if(prgg.HRrefNum)
      {
 	rect.topLeft.x -= 1;
  	rect.topLeft.y -= 1;
  	rect.extent.x  += 2;
  	rect.extent.y  += 2;
	HRWinDrawGrayRectangleFrame(prgg.HRrefNum, simpleFrame, &rect);
      }	    
      else	  
	    WinDrawGrayRectangleFrame(simpleFrame, &rect);
      if(prgg.isHiRes)
      {
        WinSetDrawWindow(oldWh);
        WinSetCoordinateSystem(kCoordinatesStandard);
      }
    }
    if(hdr.sqSel.end == sq)
    {
// Display the end selection mark
      if (prgg.isHiRes)
      {
        oldWh = WinSetDrawWindow(prgg.DispWindowH);
        WinSetCoordinateSystem(kCoordinatesNative);
      }
      if(prgg.HRrefNum)
      {
 	rect.topLeft.x -= 1;
  	rect.topLeft.y -= 1;
  	rect.extent.x  += 2;
  	rect.extent.y  += 2;
	HRWinDrawRectangleFrame(prgg.HRrefNum, simpleFrame, &rect);
      }	    
	  else  
	    WinDrawRectangleFrame(simpleFrame, &rect);
      if(prgg.isHiRes)
      {
        WinSetDrawWindow(oldWh);
        WinSetCoordinateSystem(kCoordinatesStandard);
      }
    }
  }
  if (board[sq] > 0)
  {
// Draw the piece
    x = prgg.offBoard + (c * prgg.szSq) + (prgg.szSq - wdPieces) / 2;
    y = prgg.offBoard + (r * prgg.szSq) + (prgg.szSq - PieceHeight) / 2;
    Copy2screen(((board[sq]-1)*wdPieces), PieceHeight * 2,
                wdPieces, PieceHeight, x, y, winMask);
    Copy2screen(((board[sq]-1)*wdPieces), (color[sq] == white) ? PieceHeight : 0,
                wdPieces, PieceHeight, x, y, winOverlay);
  }
}

/****************************************************************************/
void UpdateDisplay(short f, short t, short flag, short iscastle) // not S2
{
UInt16    i;
WinHandle oldWh = NULL;
TOpenChessPrefs *pp = GetPrefsPtr();

  if (flag) // complete redraw
  {
	RectangleType   rect;

    if(!pp->fullMode)
    {
// delete background (necessary because of coord.)
  	  rect.topLeft.x =
  	  rect.topLeft.y = 0;
  	  rect.extent.x =
  	  rect.extent.y = prgg.offBoard + (8 * prgg.szSq);
      if (prgg.isHiRes)
      {
        WinSetCoordinateSystem(kCoordinatesNative);
      }
      if(prgg.HRrefNum)
      {
//        rect.extent.x *= prgg.density;
//        rect.extent.y *= prgg.density;
//        HRWinEraseRectangle(prgg.HRrefNum, &rect, 0);
      }
      else
      {
        WinEraseRectangle(&rect, 0);
      }
      if (prgg.isHiRes)
      {
        WinSetCoordinateSystem(kCoordinatesStandard);
      }
// draw rectangle
  	  rect.topLeft.x =
  	  rect.topLeft.y = prgg.offBoard + wdLine - 1;
  	  rect.extent.x =
  	  rect.extent.y = (8 * prgg.szSq) - (2 * (wdLine - 1)) ;
      if (prgg.isHiRes)
      {
        oldWh = WinSetDrawWindow(prgg.DispWindowH);
        WinSetCoordinateSystem(kCoordinatesNative);
      }
      if(prgg.HRrefNum)
      {
        rect.topLeft.x -= 1;
        rect.topLeft.y -= 1;
        rect.extent.x  += 2;
        rect.extent.y  += 2;
        HRWinDrawRectangleFrame(prgg.HRrefNum, simpleFrame, &rect);
      }
      else  
      {
        WinDrawRectangleFrame(simpleFrame, &rect);
      }
      if (prgg.isHiRes)
      {
        WinSetDrawWindow(oldWh);
        WinSetCoordinateSystem(kCoordinatesStandard);
      }
// draw coordinates
  	  for(i = 0; i < 8; i++)
  	  {
        Copy2screen(offSqWhite + fullSq, i*wdLetterY,
                    wdCoordinateX, wdLetterY,
		    (hdr.reverse ? 7-i : i)*prgg.szSq + ((prgg.szSq-wdCoordinateX)/2) + prgg.offBoard, 0, winPaint);
        Copy2screen(offSqWhite + fullSq + wdCoordinateX, i*wdFigureY,
                    wdCoordinateX, wdFigureY,
		    0, (hdr.reverse ? i : 7-i)*prgg.szSq + ((prgg.szSq-wdFigureY)/2) + prgg.offBoard, winPaint);
  	  }
	}
// draw each piece
  	for (i = 0; i < 64; i++)
  		DrawPiece(i);
  }
  else
  {
    DrawPiece(f);
    DrawPiece(t);
    if (iscastle)
    {
      if (t > f)
      {
	      DrawPiece(f + 3);
	      DrawPiece(t - 1);
	  }
      else
      {
	      DrawPiece(f - 4);
	      DrawPiece(t + 1);
	  }
    }
  }
}

/****************************************************************************/
Boolean getsq(Int16 ScreenX, Int16 ScreenY, UInt16 *sq) // S2
{
  TGlobals  *pgp = GetGameGlobals();
  TGame     *gp = GetGamePtr();
	
  if(pgp->isHiRes || pgp->HRrefNum)
  {
    ScreenX *= 2; ScreenY *= 2;
  }
  ScreenX -= pgp->offBoard;
  ScreenY -= pgp->offBoard;
  if ((ScreenX < 8 * pgp->szSq) && 
      (ScreenY < 8 * pgp->szSq) && 
      (ScreenX >= 0) && (ScreenY >= 0))
  {
    *sq = 8 * (7 - (ScreenY / pgp->szSq)) + (ScreenX / pgp->szSq);
    if (gp->reverse) 
      *sq = 63 - *sq;
    return true;
  }
  else 
    return false;
}

/****************************************************************************/
void UnselectPieces(void) // S2
{
  Int16   f, t;
  TGame     *gp = GetGamePtr();


  f = gp->sqSel.start;
  t = gp->sqSel.end;
  gp->sqSel.start = gp->sqSel.end = NoSqSelected;
  DrawPiece(f);
  DrawPiece(t);
}

/****************************************************************************/
void GiveHint(void)   // S2
{
  Int16    f, t;
  TGame    *gp = GetGamePtr();
  TGlobals *pgp = GetGameGlobals();

  if(gp->hint)
  {		// Added, as there's no hint on the first move
    f = (short)(gp->hint>>8);
    t = (short)(gp->hint & 0x3F);
    UnselectPieces();
    gp->sqSel.start = f;
    gp->sqSel.end = t;
    UpdateDisplay(f,t,0,0);
  }
  else
    ShowMessage(SkipComments(11, pgp->ProgStrings));
}
/****************************************************************************/
void SwitchSides(void)      // Adapted from uxdsp.c
{
  short otherside[3]={1,0,2};	
	
  hdr.computer = otherside[hdr.computer];
  hdr.opponent = otherside[hdr.opponent];
  hdr.force = false;
  Sdepth = 0;
}

/****************************************************************************/
void ElapsedTime(short iop) // not S2
/*
   Determine the time that has passed since the search was started. If
   the elapsed time exceeds the target (ResponseTime+ExtraTime) then set
   timeout to true which will terminate the search.
*/
{
Int16           ScreenX, ScreenY, mtr;
Int32			RplEtime;
Boolean         PenDown;
Char            strDisplay[MAX_STATUS];
static char     currF, currT;
static UInt16   calcTime;

  if(iop == 1)
  {
    if(hdr.GameCnt >= 0)
      MoveString(hdr.GameCnt, 2);
    prgg.last_mtr = 0;
    currF = currT = -1;
    calcTime = 0;
  }
  et = TimGetTicks()- time0;
  if (et < 0) et = 0;
  ETnodes += 50;
  if (et > et0 || iop == 1)
    {
      RplEtime = ResponseTime+ExtraTime;
      if (et > RplEtime && Sdepth > 1) timeout = true;
      et0 = et;
      // Added : display the progress meter
      mtr = (RplEtime == 0) ? 100 : (100 * et) / RplEtime;
      if(((prgg.debugLevel == DBG_15MIN_HASH_ON) || 
          (prgg.debugLevel == DBG_15MIN_HASH_OFF)) && !iop)
      {
        if(root == &Tree[0])
	{
          if((currF != root->f) || (currT != root->t))  
          {
            currF = (char)root->f;
            currT = (char)root->t;
            calcTime = (UInt16)(et / SysTicksPerSecond());
          }
          algbr(root->f, root->t);
          StrPrintF(strDisplay, "%s=%ds(%d)D=%d.T=%d.H=%ld", 
            prgg.mvstr, calcTime, 900-((UInt16)(et / SysTicksPerSecond())),
            Cdepth, TrPnt[Cdepth+1],HashCnt);
          if(((UInt16)(et / SysTicksPerSecond())) >= 900)
            timeout = true;
          ShowMessage(strDisplay);
	}  
      }
      else if((mtr > 0) && (mtr < 100) && (mtr != prgg.last_mtr) && (iop == 0) && !prgg.Book)
	{

          StrPrintF(strDisplay, 
            SkipComments(13, prgg.ProgStrings), prgg.last_mv_string, mtr);
          ShowMessage(strDisplay);
          prgg.last_mtr = mtr;
    }
      if (iop == 1)
        {
          time0 = TimGetTicks() /*time((long *)0)*/ ; et0 = 0;
          // Added : display the progress Background
//			    Copy2screen(XOffPrgBkg, YOffPrgBkg, szSq, Prg, Xprg, Yprg, winPaint);
				  // End added
        }
      ETnodes = NodeCnt + 50;
    }
  // Added: check for user pen or kbd input. In this case, aborts current computer move.
  EvtResetAutoOffTimer();		// Reset auto off timer during computer move
  EvtGetPen(&ScreenX, &ScreenY, &PenDown);
  if ((PenDown || !EvtKeyQueueEmpty()) && (Sdepth > 1))
  {
    timeout = true;             // Abort move in case on PenDown or key pressed
//    abortedMove = true;         // signifies an aborted move due to user evt
  }
}

/****************************************************************************/
void SetTimeControl(void)   // Adapted from uxdsp.c
{
  if (hdr.TCflag)
    {
      hdr.TimeControl.moves[white] = hdr.TimeControl.moves[black] = TCmoves;
      hdr.TimeControl.clock[white] = hdr.TimeControl.clock[black] = 60*(long)TCminutes;
    }
  else
    {
      hdr.TimeControl.moves[white] = hdr.TimeControl.moves[black] = 0;
      hdr.TimeControl.clock[white] = hdr.TimeControl.clock[black] = 0;
    }
  et = 0;
  ElapsedTime(1);
}

/****************************************************************************/
void PlaySound(Int16 side, Boolean take)
{
UInt16 	i;
Tsound 	sound;
SndCommandType	cmd = {sndCmdFreqDurationAmp, 0, 0, 0 ,0};
Tsound opponentMvSnd = {1, {{440, 40}}};
Tsound computerMvSnd = {1, {{660, 40}}};
Tsound opponentTkSnd = {2, {{440, 80}, {527, 40}}};
Tsound computerTkSnd = {2, {{660, 80}, {527, 40}}};
Tsound chessSnd = {7, {{527, 200}, {527, 200}, {527, 200}, {702, 200},
											 {0, 100}, {527, 200}, {702, 400}}};
Tsound mateSnd = {15, {{527, 400}, 	// Mozart's funeral march
											 {0, 100},
                       {527, 400},
                       {527, 200},
                       {527, 200},
                       {0, 400},
                       {626,400},
                       {591, 200},
                       {591, 200},
                       {0, 100},
                       {527, 200},
                       {527, 200},
                       {0, 100},
                       {470, 200},
                       {527, 200}}};
TOpenChessPrefs *pp = GetPrefsPtr();
UInt16 SndLevel;
	if (!pp->beep) return;
	SndLevel = (UInt16)PrefGetPreference(prefGameSoundVolume);
	if (side == white) sound = take ? opponentTkSnd : opponentMvSnd;
	if (side == black) sound = take ? computerTkSnd : computerMvSnd;
	if (SqAtakd(PieceList[black][0],white) ||		  // Chess attack
  		SqAtakd(PieceList[white][0],black)) sound = chessSnd;
  if (hdr.endgame) sound = mateSnd;		// Mate sound

	for (i = 0; i < sound.nNotes; i++) {
  	cmd.param1 = (Int32)sound.param123[i][0];
  	cmd.param2 = sound.param123[i][1];
  	cmd.param3 = SndLevel;
    if (cmd.param1 == 0) 			// Wait, no sound
    		SysTaskDelay((SysTicksPerSecond() * cmd.param2) / 1000);
  	else SndDoCmd (NULL, &cmd, 0);
		SysTaskDelay((SysTicksPerSecond() * 30) / 1000);
    }
}

/****************************************************************************/
void DisplayStatus(void) //S2
{
  char  s[MAX_STATUS],
        *tmp = NULL;
  short i;      
  TGame    *gp = GetGamePtr();
  TGlobals *pgp = GetGameGlobals();

  if((pgp->debugLevel == DBG_15MIN_HASH_OFF) ||
     (pgp->debugLevel == DBG_15MIN_HASH_ON))
    return;

  SetLibText();
  
  if(gp->GameCnt >= 0)	
  {
// display move text with move number
    tmp = MoveString(gp->GameCnt, 2);
    
    if(pgp->Book && (pgp->LibText != NULL) && gp->TrainerMode && *pgp->LibText)
    {
      StrPrintF(s, "%s ", tmp);
      i = StrLen(s);
      tmp = pgp->LibText;
      while((i < MAX_STATUS-2) && *tmp)
      {
	    s[i++] = *tmp++; 
      } 
      s[i] = '\0';
    }
    else if(gp->endgame)
      StrPrintF(s, "%s %s", tmp, SkipComments(gp->endgame, pgp->ProgStrings));
    else
      StrCopy(s, tmp);  
	ShowMessage(s);
  }
  else if(gp->endgame)
  {
    ShowMessage(SkipComments(gp->endgame, pgp->ProgStrings));  
  }
  else if(pgp->Book && (pgp->LibText != NULL) && gp->TrainerMode && *pgp->LibText)
  {
	ShowMessage(pgp->LibText);
  }
  else
  {
	ShowMessage("");
  }
}

/****************************************************************************/
void OutputMove(void)   // not S2
{
//  if (abortedMove)
//    return;  // Added, do not draw anything if the move is
             // interrupted by a user input

  if (root->flags & epmask)
    UpdateDisplay(0,0,1,0);
  else
  {
    UnselectPieces();
    hdr.sqSel.start = root->f;
    hdr.sqSel.end = root->t;
    UpdateDisplay(root->f,root->t,0,root->flags & cstlmask);
  }
  DisplayStatus();
}

/****************************************************************************/
void Undo(void)     // Copied from uxdsp.c, Undo()
/*
   Undo the most recent half-move.
*/
{
  short f,t;
  short otherside[3]={1,0,2};	

  if(hdr.GameCnt < 0) 
    return; 
  f = GameList[hdr.GameCnt].gmove>>8;
  t = GameList[hdr.GameCnt].gmove & 0xFF;
  hdr.sqSel.start = f; 
  hdr.sqSel.end = t;
  if (GameList[hdr.GameCnt].flags & cstlmask)
    castle(color[t],f,t,2);
  else
    {
      if(GameList[hdr.GameCnt].flags & epmask)
        EnPassant(otherside[color[t]],f,t,2);
      board[f] = board[t]; 
      color[f] = color[t];
      if(GameList[hdr.GameCnt].flags & promote)
        board[f] = pawn;
      hdr.sqSel.start = f; 
      hdr.sqSel.end = t;
      if(!(GameList[hdr.GameCnt].flags & epmask))
        board[t] = GameList[hdr.GameCnt].capt_piece;
      else
        board[t] = no_piece;
      if(board[t])
        color[t] = otherside[color[f]];
      else      
        color[t] = neutral;
      if (board[f] == king)
        --hdr.kingmoved[color[f]];
// check whether rook moved from its original position
      if (board[f] == rook)
      {
	      if (color[f] == white)
	      {
		      if (f == 0)
		        --hdr.lrookmoved[color[f]];
		      else if (f == 7)
		        --hdr.hrookmoved[color[f]];
	      }
	      else // black
	      {
		      if (f == 56)
		        --hdr.lrookmoved[color[f]];
		      else if (f == 63)
		        --hdr.hrookmoved[color[f]];
	      }
      }
    }
  if (hdr.TCflag)
    ++hdr.TimeControl.moves[color[f]];
  hdr.GameCnt--;
  if(hdr.GameCnt >= 0) 
    hdr.hint = GameList[hdr.GameCnt].hint;
  else
    GetOpenings(); 
  hdr.endgame = 0;
  Sdepth = 0;
}

/****************************************************************************/
void Back(void) // S2
/*
   One step back
*/
{
  short otherside[3]={1,0,2};	
  TGame *gp = GetGamePtr();
  TGlobals *pgp = GetGameGlobals();
    
  if(gp->GameCnt < 0) 
    return;
  
  if(gp->bothsides)
  {
    if(gp->endgame)
    {
      Undo();
      Undo();
    }
    else // do not disturb auto-playing
    {
      return; 
    }
  }
// no Back if computer did first move
  else if(!gp->force && (gp->GameCnt == 0) && (gp->player == gp->opponent))
  {
    return;
  }
// only one move back if Human vs. Human
  else if(gp->force)
  {
    Undo();
    gp->player = gp->computer;
    gp->computer = gp->opponent;
    gp->opponent = otherside[gp->computer];
  }
// if last move made by Human and game is over
  else if(gp->endgame && (gp->player != gp->opponent))
  {
    Undo();
    gp->player = gp->opponent;
  }
// otherwise 2 moves back
  else
  {
    Undo();
    Undo();
  }
  pgp->mvstr[0] = '\0';
  UpdateDisplay(0,0,1,0);
  DisplayStatus();
  InitializeStats();
  return;
}

/****************************************************************************/
void ComputerPlay(short side)
{
  Int16   ScreenX, ScreenY;
  Boolean PenDown;

  if ((prgg.DispWindowH == WinGetDrawWindow()) && prgg.playingMode)
  {
// wait
	do
      EvtGetPen(&ScreenX, &ScreenY, &PenDown);
    while (PenDown);

    ElapsedTime(1);
    SelectMove(side,1);  // Gnuchess.c, main()
/*    if (abortedMove)
    {  // Restart search
      Undo();
      abortedMove = false;
      hdr.player = side;
    }
    else*/
      PlaySound(side, (root->flags & capture));
  }
}

/****************************************************************************/
static Boolean ApplicationHandleEvent(EventPtr event) // S2
{
UInt16 formID;
FormPtr form;
Boolean handled = false;
//TGlobals *pgp = GetGameGlobals();

  switch (event->eType)
  {          // Application event loop
    case frmLoadEvent:
// Handle form load events
        formID = event->data.frmLoad.formID;
        form = FrmInitForm(formID);
        FrmSetActiveForm(form);
	while(formID >= frmMain+resLF)
	  formID -= resLF;
//        pgp->playingMode = false;
// Set event handler
        switch (formID)
        {
          case frmPrefs:
            FrmSetEventHandler(form, (FormEventHandlerPtr) frmPrefs_HandleEvent);
            break;
          case frmOptions:
            FrmSetEventHandler(form, (FormEventHandlerPtr) frmOptions_HandleEvent);
            break;
          case frmEditBrd:
            FrmSetEventHandler(form, (FormEventHandlerPtr) frmEditBrd_HandleEvent);
/*            WinSetConstraintsSize(FrmGetWindowHandle(form), 
              160, 160, 160, 160, 160, 160);*/
            break;
          case frmAbout:
            FrmSetEventHandler(form, (FormEventHandlerPtr) frmAbout_HandleEvent);
            break;
          case frmMessage:
            FrmSetEventHandler(form, (FormEventHandlerPtr) frmMessage_HandleEvent);
            break;
          case frmPieces:
            FrmSetEventHandler(form, (FormEventHandlerPtr) frmPieces_HandleEvent);
            break;
          case frmPiecesTaken:
            FrmSetEventHandler(form, (FormEventHandlerPtr) frmPiecesTaken_HandleEvent);
            break;
          case frmDatabase:
            FrmSetEventHandler(form, (FormEventHandlerPtr) frmDatabase_HandleEvent);
            break;
          case frmMain:
          case frmMainFull:
            FrmSetEventHandler(form, (FormEventHandlerPtr) frmMain_HandleEvent);
/*            WinSetConstraintsSize(FrmGetWindowHandle(form), 
              160, 160, 160, 160, 160, 160);*/
//            pgp->playingMode = true;
            break;
        }
        handled = true;
        break;

    default:
      break;
    }

  return handled;
}

/****************************************************************************/
Boolean DoEvents(void) // S2
{
UInt16 error;
EventType event;
TOpenChessPrefs *pp = GetPrefsPtr();
TGame           *gp = GetGamePtr();
TGlobals        *pgp = GetGameGlobals();

  do
  {
    if ((gp->bothsides || (gp->player == gp->computer)) && !gp->endgame)
      EvtGetEvent(&event, SysTicksPerSecond() / 4); // 250 ms
    else
      EvtGetEvent(&event, evtWaitForever);

// Detect Grafitti to Screen Stroke (back from Fullscreen)
    if (pp->fullMode)
    {
      if ((event.eType == penDownEvent) &&
	      (event.screenY >= 160))
	  {
          pgp->fullModeOff = true;
      }
      else
      {
        if ((event.eType == penUpEvent) && pgp->fullModeOff)
        {
          if (FrmDispatchEvent(&event))
            continue;
        }
      }
    }

// Standard event handling
    if (!SysHandleEvent(&event))
    {
      if (!MenuHandleEvent(0, &event, &error))
      {
        if (!ApplicationHandleEvent(&event))
          FrmDispatchEvent(&event);
      }   // if (!MenuHandleEvent(...
    }     // if (!SysHandleEvent(...
  }       // while (EvtEventAvail...
  while ((EvtEventAvail() || EvtSysEventAvail(false)) && (event.eType != appStopEvent));
  return(event.eType == appStopEvent);
}

/****************************************************************************/
static void EventLoop(void)     // does almost the same as in gnuchess.c, main()
{
  while (!DoEvents()) {
    if ((hdr.player == hdr.opponent) && hdr.bothsides && !hdr.endgame )
      ComputerPlay(hdr.opponent);
    if (DoEvents())
      return;
    if ((hdr.player == hdr.computer) && !(hdr.endgame || hdr.force))
      ComputerPlay(hdr.computer);
    }
}

/****************************************************************************/
static void StopApplication(void)
{
  FreeBookMem();
  FreeHashMem();
  FreeLibMem();

  if(prgg.OffscreenWinH)
    WinDeleteWindow(prgg.OffscreenWinH, false);  // Delete offscreenwindow

  if(prgg.v3bmp)
    BmpDelete((BitmapType *)prgg.v3bmp);

  if(prgg.HRrefNum)
  {
    if(prgg.HRversion < HR_VERSION_SUPPORT_FNTSIZE) 
      HRWinScreenMode(prgg.HRrefNum, winScreenModeSetToDefaults, NULL, NULL, NULL, NULL);
    HRClose(prgg.HRrefNum);
  }
// Save the preferences and last game
  SaveGameToDB(prefs.DbGameIndex);
  SavePrefs(&prefs);
  DmCloseDatabase (prgg.OpenChessDB);

  if(history)
    MemPtrFree(history);
  if(prgg.ProgStrings)
    MemPtrFree(prgg.ProgStrings);
  FrmCloseAllForms();
}

/****************************************************************************/
UInt32 PilotMain(UInt16 cmd, void *cmdPBP, UInt16 launchFlags)
{
int error;

  if (cmd == sysAppLaunchCmdNormalLaunch)
  {
    error = StartApplication();
    if (error)
      return error;
    EventLoop();
    StopApplication();
  }
  return 0;
}

/****************************************************************************/
Boolean frmMain_HandleEvent(EventPtr event) // S2
{
FormPtr form;
Boolean handled = false;
UInt16  sq;
EventType newEvent;
TOpenChessPrefs *pp = GetPrefsPtr();
TGame           *gp = GetGamePtr();
TGlobals        *pgp = GetGameGlobals();

  form = FrmGetActiveForm();
  switch (event->eType)
  {
    case frmOpenEvent:
      SetFullMode(event->data.frmOpen.formID == frmMainFull);
      FrmSetMenu(form, mnufrmMain+(pp->Langu*resLF));
    case winDisplayChangedEvent:
      newEvent.eType = firstUserEvent;
      EvtAddUniqueEventToQueue(&newEvent, 0, false);
    case frmUpdateEvent:
      FrmDrawForm(form);
      pgp->DispWindowH = WinGetDrawWindow(); // Used then to detect if the menu is called
      handled = true;
      break;

    case winEnterEvent:
      if(pgp->DispWindowH == WinGetDrawWindow())
      {
        newEvent.eType = firstUserEvent;
        EvtAddUniqueEventToQueue(&newEvent, 0, false);
        handled = true;
      }
      break;

    case firstUserEvent:
      pgp->playingMode = true;
      UpdateDisplay(0, 0, true, false);
      if(!pp->fullMode)
        DrawButton(form, bmpButtonsPlay, btnNew);
      DisplayStatus();
      handled = true;
      break;

    case keyDownEvent:
	  if (event->data.keyDown.chr == vchrPageUp)
	  {
        Back();
        handled = true;
      }
      break;

    case penUpEvent:
// detect back from Fullscreen
      if (pgp->fullModeOff)
      {
         if (event->screenY < 160)
         {
		    FrmGotoForm(frmMain);
            handled = true;
         }
      }
      pgp->fullModeOff = false;
      break;

    case penDownEvent:
      if(hdr.bothsides)
        break;
      if(!gp->force && (gp->player == gp->computer))
        break;
      if(getsq(event->screenX, event->screenY, &sq))
      {
	if(color[sq] == gp->opponent)
	{ // selection of an opponent piece ?
          if(gp->sqSel.start >= 0)
	    UnselectPieces();   // First, unselect previous selections
          gp->sqSel.start = sq;       // Select new
          DrawPiece(sq);
        }
        else
	  if((gp->sqSel.start >= 0) && (gp->sqSel.end == NoSqSelected))
	  { // try a move ?
            gp->sqSel.end = sq;
            gp->player = gp->opponent;
            algbr(gp->sqSel.start,gp->sqSel.end);
            StrCopy(pgp->man_mv,pgp->mvstr);  // uxdsp.c, InputCommand()
            ManualMove();
	  }
	  handled = true;
      }
      else
        if((event->screenX <= (X_TEXT+EX_TEXT)) && (event->screenY > Y_TEXT) && 
	    !pp->fullMode && pgp->Book && (pgp->LibText != NULL) && gp->TrainerMode)
          FrmPopupForm(frmMessage+(pp->Langu*resLF));
      break;

    case ctlSelectEvent:		// transform buttons into menu commands
    	switch(event->data.ctlSelect.controlID) {
      	case btnNew:			event->data.menu.itemID = mnuNew; break;
        case btnOptions: 	event->data.menu.itemID = mnuOptions; break;
      	case btnLoad: 		event->data.menu.itemID = mnuLoad; break;
      	case btnEdit:			event->data.menu.itemID = mnuEdit; break;
      	case btnUndo:			event->data.menu.itemID = mnuUndo; break;
      	case btnHint:			event->data.menu.itemID = mnuHint; break;
      	case btnReverse:	event->data.menu.itemID = mnuReverse; break;
      	case btnSwitch:		event->data.menu.itemID = mnuSwitchSides; break;
      	case btnMissingPieces:		event->data.menu.itemID = mnuMissingPieces; break;
      	case btnFullscreen:		event->data.menu.itemID = mnuToogleSize; break;
		default:					event->data.menu.itemID = 0;
      	}

    case menuEvent:
      switch(event->data.menu.itemID) {
        case mnuNew:
          pgp->IsNewGame = true;
          FrmPopupForm(frmOptions+(pp->Langu*resLF));
          handled = true;
          break;

        case mnuOptions:
          pgp->IsNewGame = false;
          FrmPopupForm(frmOptions+(pp->Langu*resLF));
          handled = true;
          break;

        case mnuPrefs:
          FrmGotoForm(frmPrefs+(pp->Langu*resLF));
          handled = true;
          break;

        case mnuEdit:
     	  gp->TrainerMode = false;
     	  gp->flags |= game_flag_edit;
          pgp->Book = NULL;
	  pgp->last_mv_string[0] = '\0';
          FrmGotoForm(frmEditBrd);
          handled = true;
          break;

        case mnuLoad:
          FrmGotoForm(frmDatabase+(pp->Langu*resLF));
          handled = true;
          break;

        case mnuSave:
          SaveGameToDB(0);
          pp->DbGameIndex = SaveGameToDB(pp->DbGameIndex);
          handled = true;
          break;

        case mnuDelete:
	  if (pp->DbGameIndex)
	  {
            if (DmQueryRecord(pgp->OpenChessDB, pp->DbGameIndex))
	    {
              SaveGameToDB(0);
              DmRemoveRecord(pgp->OpenChessDB, pp->DbGameIndex);
              pp->DbGameIndex = 0;
              if (1 == DmNumRecords(pgp->OpenChessDB))
                pp->SaveIndex = 1;
	    }
	  }
          handled = true;
          break;

        case mnuUndo:
        	MenuEraseStatus(NULL); 	// Clears the menu command bar.
          Back();
          handled = true;
          break;

        case mnuHint:
        	MenuEraseStatus(NULL); 	// Clears the menu command bar.
          GiveHint();
          handled = true;
          break;

        case mnuReverse:
          gp->reverse = !gp->reverse;
          UpdateDisplay(0,0,1,0);
          handled = true;
          break;

        case mnuSwitchSides:
          SwitchSides();
          handled = true;
          break;

        case mnuHelp:
          FrmHelp(HelpString+(pp->Langu*resLF));
          handled = true;
          break;

        case mnuAbout:
          FrmPopupForm(frmAbout+(pp->Langu*resLF));
          handled = true;
          break;

        case mnuMissingPieces:
          FrmPopupForm(frmPiecesTaken+(pp->Langu*resLF));
          handled = true;
          break;

	case mnuToogleSize:
          if (pp->fullMode)
		    FrmGotoForm(frmMain);
          else
          {
			if (pp->fullModeMsg)
			{
				if (FrmAlert(AlertFullModeMsg+(pp->Langu*resLF)) != 0) // do NOT remind again
				{
					pp->fullModeMsg = false;
				}
			}
	        FrmGotoForm(frmMainFull);
		  }
          handled = true;
          break;
          
        case mnuExport:
          PGNExport();
          handled = true;
          break;
        }

    default:
      break;
  }

  return handled;
}

/****************************************************************************/
Boolean frmEditBrd_HandleEvent(EventPtr event) // not S2
{
FormPtr form;
Boolean handled = false;
UInt16  sq;
EventType newEvent;
short otherside[3]={1,0,2};	
TOpenChessPrefs *pp = GetPrefsPtr();
TGame           *gp = GetGamePtr();
TGlobals        *pgp = GetGameGlobals();

  form = FrmGetActiveForm();
  switch (event->eType)
  {
    case frmOpenEvent:
      FrmSetMenu(form, mnufrmEditBrd+(pp->Langu*resLF));
      pgp->playingMode = false;
      pp->editMode = true;
      SetFullMode(false);
      gp->sqSel.start = 
      gp->sqSel.end = NoSqSelected;
    case winDisplayChangedEvent:
      newEvent.eType = firstUserEvent;
      EvtAddUniqueEventToQueue(&newEvent, 0, false);
      newEvent.eType = firstUserEvent+2;
      EvtAddUniqueEventToQueue(&newEvent, 0, false);
    case frmUpdateEvent:
      FrmDrawForm(form);
      pgp->DispWindowH = WinGetDrawWindow(); // Used then to detect if the menu is called
      handled = true;
      break;

    case winEnterEvent:
      if(pgp->DispWindowH == WinGetDrawWindow())
      {
        newEvent.eType = firstUserEvent;
        EvtAddUniqueEventToQueue(&newEvent, 0, false);
        handled = true;
      }
      break;

    case firstUserEvent:
      UpdateDisplay(0, 0, true, false);
      DrawButton(form, bmpButtonsEdit, btnClear);
      handled = true;
      break;

    case firstUserEvent+1:
      CheckRules(event->data.generic.datum[0]);
      handled = true;
      break;

    case firstUserEvent+2:
      ShowMessage(SkipComments(24+hdr.player, pgp->ProgStrings));
      handled = true;
      break;

    case penDownEvent:    // select piece
      if (getsq(event->screenX, event->screenY, &sq))
      { // a square is tapped
		if (hdr.sqSel.start == sq)
		  UnselectPieces();	// Dual tap : unselect square
		else if (board[hdr.sqSel.start] != no_piece)
		{ // if a piece is previously selected
          board[sq] = board[hdr.sqSel.start];
          color[sq] = color[hdr.sqSel.start]; // move it there
          board[hdr.sqSel.start] = no_piece;
          color[hdr.sqSel.start] = neutral;
          InitializeStats();
          UpdateDisplay(hdr.sqSel.start,sq,0,false);
          UnselectPieces();
          newEvent.eType = firstUserEvent+1;
          newEvent.data.generic.datum[0] = sq;
          EvtAddUniqueEventToQueue(&newEvent, 0, false);
        }
		else
		{        	       // Select new
          UnselectPieces();
          hdr.sqSel.start = sq;
          DrawPiece(sq);
   	    }
        handled = true;
      }
      break;

    case ctlSelectEvent:		// transform buttons into menu commands
    	switch(event->data.ctlSelect.controlID) {
      	case btnClear:			event->data.menu.itemID = mnuClear; break;
        case btnAdd: 				event->data.menu.itemID = mnuAdd; break;
      	case btnRemove: 		event->data.menu.itemID = mnuRemove; break;
      	case btnReverse:	event->data.menu.itemID = mnuReverse; break;
      	case btnSwitch:		event->data.menu.itemID = mnuSwitchSides; break;
      	case btnMissingPieces:		event->data.menu.itemID = mnuMissingPieces; break;
      	case btnExit: 			event->data.menu.itemID = mnuClose; break;
				default:						event->data.menu.itemID = 0;
      	}

		case menuEvent:
      switch(event->data.menu.itemID) {
      	case mnuClose:
      	    if (CheckBoardEdit())
	    {
              pp->editMode = false;
              UnselectPieces();
              BackToMain();
	    }
  	    handled = true;
    	    break;

        case mnuClear:
          if (FrmAlert(ConfClearBd+(pp->Langu*resLF)) == 0) 
          {	// OK pressed
            for (sq = 0; sq < 64; sq++) 
            {		
              board[sq] = no_piece; 
              color[sq] = neutral;
            }
            InitializeStats();
	        UpdateDisplay(0,0,1,0);
          }
          handled = true;
          break;

        case mnuAdd:
	  if(hdr.sqSel.start != NoSqSelected) // a square was selected
            FrmPopupForm(frmPieces+(pp->Langu*resLF));
          else 
	    FrmAlert(AlertSelectSq+(pp->Langu*resLF));
          handled = true;
          break;

        case mnuRemove:
	        if (hdr.sqSel.start != NoSqSelected) 
	        { // a square was selected
	          board[hdr.sqSel.start] = no_piece; 
	          color[hdr.sqSel.start] = neutral;
              InitializeStats();
  	          DrawPiece(hdr.sqSel.start);
            }
    	    handled = true;
          break;

        case mnuReverse:
          hdr.reverse = !hdr.reverse;
          UpdateDisplay(0,0,1,0);
          handled = true;
          break;

        case mnuSwitchSides:
          hdr.player = otherside[hdr.player];
          if (hdr.force)
          {
            hdr.computer = hdr.opponent;
            hdr.opponent = otherside[hdr.computer]; 
	  }  
          newEvent.eType = firstUserEvent+2;
          EvtAddUniqueEventToQueue(&newEvent, 0, false);
          handled = true;
          break;

        case mnuHelpEdit:
        	FrmHelp(StrHelpEdit+(pp->Langu*resLF));
          handled = true;
          break;

        case mnuMissingPieces:
          FrmPopupForm(frmPiecesTaken+(pp->Langu*resLF));
          handled = true;
          break;

        case mnuAbout:
          FrmPopupForm(frmAbout+(pp->Langu*resLF));
          handled = true;
          break;
        }

    default:
      break;
    }

    return handled;
}

/****************************************************************************/
Boolean frmPrefs_HandleEvent(EventPtr event) //S2
{
FormPtr form;
ListPtr ListP;
Boolean handled = false;
UInt16	i, selW, selB;
DmSearchStateType searchState;
UInt16 dbCard;
LocalID dbId;
TOpenChessPrefs *pp = GetPrefsPtr();
TGlobals  *pgp = GetGameGlobals();

  form = FrmGetActiveForm();
  switch (event->eType)
  {
    case frmOpenEvent:
      pgp->playingMode = false;

      ListP = GetCtrlPtr(form, lstLangu);
      LstSetSelection(ListP, pp->Langu);
      
      CtlSetLabel(GetCtrlPtr(form, trgLangu), LstGetSelectionText(ListP, pp->Langu));

      CtlSetValue(GetCtrlPtr(form, chkSound), pp->beep);

      CtlSetValue(GetCtrlPtr(form, chkFullMsg), pp->fullModeMsg);

      CtlSetValue(GetCtrlPtr(form, chkNormGames), pp->NormGames);

      CtlSetValue(GetCtrlPtr(form, chkOnlyTrainer), pp->OnlyTrainer);

      selW = selB = noListSelection;
      if(DmGetNextDatabaseByTypeCreator(true, &searchState,
           OpenChessOLType, sysFileCOpenChess, false, &dbCard, &dbId) != dmErrCantFind)
      {
        do
        {
	      if(dbCard == 0)
            pgp->nLib++;
        }
        while(DmGetNextDatabaseByTypeCreator(false, &searchState,
                OpenChessOLType, sysFileCOpenChess, false, &dbCard, &dbId) != dmErrCantFind);
      }
      if(pgp->nLib != 0)
      {
        pgp->LibNames = MemPtrNew(sizeof(char *) * pgp->nLib);
        CRASHIF(!pgp->LibNames, "No memory.");

        i = 0;
        if(DmGetNextDatabaseByTypeCreator(true, &searchState,
             OpenChessOLType, sysFileCOpenChess, false, &dbCard, &dbId) != dmErrCantFind)
        {
          do
          {
	        if(dbCard == 0)
	        {
	          pgp->LibNames[i] = MemPtrNew(dmDBNameLength);
              CRASHIF(!pgp->LibNames[i], "No memory.");
              DmDatabaseInfo(dbCard, dbId, pgp->LibNames[i], NULL, NULL,
                                NULL, NULL, NULL, NULL, NULL, NULL,
                                NULL, NULL);

              if(StrLen(pp->OLibWhite) &&
                 !StrCompare(pp->OLibWhite, pgp->LibNames[i]))
                selW = i;

              if(StrLen(pp->OLibBlack) &&
                 !StrCompare(pp->OLibBlack, pgp->LibNames[i]))
                selB = i;

	          i++;
            }  

          } 
          while((DmGetNextDatabaseByTypeCreator(false, &searchState,
                   OpenChessOLType, sysFileCOpenChess, false, &dbCard, &dbId) != dmErrCantFind) &&
            (i <= pgp->nLib));
        }
      }

      ListP = GetCtrlPtr(form, lstLibWhite);
      LstSetListChoices(ListP, pgp->LibNames, pgp->nLib);
      LstSetHeight (ListP, pgp->nLib < MAX_LIB_LIST ? pgp->nLib : MAX_LIB_LIST);
      LstSetSelection(ListP, selW);
      CtlSetLabel(GetCtrlPtr(form, trgLibWhite), LstGetSelectionText(ListP, selW));

      ListP = GetCtrlPtr(form, lstLibBlack);
      LstSetListChoices(ListP, pgp->LibNames, pgp->nLib);
      LstSetHeight (ListP, pgp->nLib < MAX_LIB_LIST ? pgp->nLib : MAX_LIB_LIST);
      LstSetSelection(ListP, selB);
      CtlSetLabel(GetCtrlPtr(form, trgLibBlack), LstGetSelectionText(ListP, selB));

      FrmDrawForm(form);
      handled = true;
      break;

    case ctlSelectEvent:
      if (event->data.ctlSelect.controlID == btnOK)
      {
        handled = true;

        pp->Langu = LstGetSelection(GetCtrlPtr(form, lstLangu));

        pp->beep = CtlGetValue(GetCtrlPtr(form, chkSound));

        pp->fullModeMsg = CtlGetValue(GetCtrlPtr(form, chkFullMsg));

        i = LstGetSelection(GetCtrlPtr(form, lstLibWhite));
        if(i != noListSelection)
        {
          StrCopy(pp->OLibWhite, pgp->LibNames[i]);
        }
        else
        {
          *pp->OLibWhite = '\0';
        }

        i = LstGetSelection(GetCtrlPtr(form, lstLibBlack));
        if(i != noListSelection)
        {
          StrCopy(pp->OLibBlack, pgp->LibNames[i]);
        }
        else
        {
          *pp->OLibBlack = '\0';
        }
        
        pp->NormGames = CtlGetValue(GetCtrlPtr(form, chkNormGames));

        pp->OnlyTrainer = CtlGetValue(GetCtrlPtr(form, chkOnlyTrainer));

      }
      else if(((event->data.ctlSelect.controlID == trgLibWhite) ||
               (event->data.ctlSelect.controlID == trgLibBlack)) &&
              !pgp->nLib)
      {
	    FrmAlert(AlertNoBooks+(pp->Langu*resLF));
        handled = true;
	break;
      }
      else if(event->data.ctlSelect.controlID == btnCancel)
      {
        handled = true;
      }
      if (handled)
      {
        FreeLibMem();
        GetProgStrings();
        BackToMain();
      }
      break;

    default:
      break;
  }
  return handled;
}

/****************************************************************************/
Boolean frmOptions_HandleEvent(EventPtr event) //S2
{
FormPtr form;
ListPtr ListP;
Boolean s, b, f, tm,
        back, handled = false;
	
UInt16  lst, l, c, d, slen;
Char    *string;
MemHandle old, h;
FieldPtr fldP;
TOpenChessPrefs *pp = GetPrefsPtr();
TGame           *gp = GetGamePtr();
TGlobals        *pgp = GetGameGlobals();

  form = FrmGetActiveForm();

  switch (event->eType)
  {
    case frmOpenEvent:
      pgp->playingMode = false;

      l = gp->Level;
      lst = 1;
      while ((l * 2) >= (1 << lst))
        lst++;
      lst--;
      ListP = GetCtrlPtr(form, lstCompTime);
      LstSetSelection(ListP, lst);
      LstMakeItemVisible(ListP, lst);
      CtlSetLabel(GetCtrlPtr(form, trgCompTime), LstGetSelectionText(ListP, lst));

      ListP = GetCtrlPtr(form, lstDither);		// Dither
      d = gp->dither / 20;
      LstSetSelection(ListP, d); LstMakeItemVisible(ListP, d);
      CtlSetLabel(GetCtrlPtr(form, trgDither), LstGetSelectionText(ListP, d));

      ListP = GetCtrlPtr(form, lstContempt);		// Contempt
      c = gp->contempt / 20;
      LstSetSelection(ListP, c); LstMakeItemVisible(ListP, c);
      CtlSetLabel(GetCtrlPtr(form, trgContempt), LstGetSelectionText(ListP, c));
      
      if (gp->bothsides)
      {
        CtlSetValue(GetCtrlPtr(form, btnWhiteCom), true);
        CtlSetValue(GetCtrlPtr(form, btnBlackCom), true);
        CtlSetValue(GetCtrlPtr(form, btnWhiteHum), false);
        CtlSetValue(GetCtrlPtr(form, btnBlackHum), false);
      }
      else if (gp->force)
      {
        CtlSetValue(GetCtrlPtr(form, btnWhiteHum), true);
        CtlSetValue(GetCtrlPtr(form, btnBlackHum), true);
        CtlSetValue(GetCtrlPtr(form, btnWhiteCom), false);
        CtlSetValue(GetCtrlPtr(form, btnBlackCom), false);
      }
      else if (gp->opponent == white)
      {
        CtlSetValue(GetCtrlPtr(form, btnWhiteHum), true);
        CtlSetValue(GetCtrlPtr(form, btnBlackCom), true);
        CtlSetValue(GetCtrlPtr(form, btnBlackHum), false);
        CtlSetValue(GetCtrlPtr(form, btnWhiteCom), false);
      }
      else
      {
        CtlSetValue(GetCtrlPtr(form, btnBlackHum), true);
        CtlSetValue(GetCtrlPtr(form, btnWhiteCom), true);
        CtlSetValue(GetCtrlPtr(form, btnWhiteHum), false);
        CtlSetValue(GetCtrlPtr(form, btnBlackCom), false);
      }
      CtlSetValue(GetCtrlPtr(form, chkOTraining), gp->TrainerMode);

      h = MemHandleNew( MAX_NAME + 1 );
      string = MemHandleLock( h );

      if (pgp->IsNewGame)
      {
        FrmCopyTitle (form, SkipComments(14, pgp->ProgStrings));
      }
      else
      {
        FrmCopyTitle(form, SkipComments(15, pgp->ProgStrings));
      }
      
//      if(!pgp->Book && !pgp->IsNewGame)
//        FrmHideObject(form, FrmGetObjectIndex (form, chkOTraining));

      if ((pgp->IsNewGame) || (!StrLen(gp->Name) && !pgp->IsNewGame))
      {
        StrPrintF(string, SkipComments(16, pgp->ProgStrings), pp->SaveIndex);
      }
      else
      {
        StrCopy(string, gp->Name);
        CtlSetValue(GetCtrlPtr(form, chkSave), true);
      }
      MemHandleUnlock( h );
      fldP = GetCtrlPtr( form, fldName );
      old = FldGetTextHandle( fldP );
      FldSetTextHandle( fldP, h );
      if( old != NULL )
        MemHandleFree( old );
      FrmDrawForm(form);
      handled = true;
      break;

    case ctlSelectEvent:
      back = false;
      if (event->data.ctlSelect.controlID == btnOK)
      {
        handled = true;

        b = (CtlGetValue(GetCtrlPtr(form, btnBlackCom)) &&
            CtlGetValue(GetCtrlPtr(form, btnWhiteCom)));
        f = (CtlGetValue(GetCtrlPtr(form, btnBlackHum)) &&
            CtlGetValue(GetCtrlPtr(form, btnWhiteHum)));

        tm = CtlGetValue(GetCtrlPtr(form, chkOTraining));

        if (tm && (b || f))
	  FrmAlert(AlertErrTrainer+(pp->Langu*resLF));
        else
        {
          fldP = GetCtrlPtr( form, fldName );
          if (NULL !=(string = FldGetTextPtr( fldP )))
            slen = StrLen( string );
	  else
	    slen = 0;
	  s = CtlGetValue(GetCtrlPtr(form, chkSave));
          if ((0 == slen) && s)
	  {
	    FrmAlert(AlertSaveGame+(pp->Langu*resLF));
	  }
	  else
	  {
            back = true;

// Save old Game
	    SaveGameToDB(pp->DbGameIndex);

        if (pgp->IsNewGame)
        {
          NewGame();
	      gp->player = gp->opponent;
          gp->sqSel.start = gp->sqSel.end = NoSqSelected;
	      gp->TrainerMode = tm;
	      *gp->Name = '\0';

          if (CtlGetValue(GetCtrlPtr(form, btnBlackHum)) &&
              CtlGetValue(GetCtrlPtr(form, btnWhiteCom)))
	      {
	        SwitchSides();
  	        gp->reverse = !gp->reverse;
	      }
	      GetOpenings();
        }
        else
        {
	      if(gp->TrainerMode)  
	        gp->TrainerMode = tm;
          if(!(b || f) &&
             (gp->opponent == white) &&
             CtlGetValue(GetCtrlPtr(form, btnBlackHum)))
          {
            SwitchSides();
          }
        }
         
        gp->bothsides = b;
	    gp->force = f;

	    gp->dither = 20 * LstGetSelection(GetCtrlPtr(form, lstDither));
	    gp->contempt = 20 * LstGetSelection(GetCtrlPtr(form, lstContempt));

        gp->Level = (1 << LstGetSelection(GetCtrlPtr(form, lstCompTime))) >> 1;

	    if (s)
	    {
	      if(pgp->IsNewGame || !pp->DbGameIndex)
	      {
	        pp->SaveIndex++;
                pp->DbGameIndex = dmMaxRecordIndex;
	      }
	      StrCopy(gp->Name, string);
              pp->DbGameIndex = SaveGameToDB(pp->DbGameIndex);
	    }
	    else
	    {
	      pp->DbGameIndex = 0;
	    }
	  }
        }
      }
      if (event->data.ctlSelect.controlID == btnCancel)
        back = handled = true;
      if((event->data.ctlSelect.controlID == chkOTraining) &&
         !pgp->IsNewGame &&
         !gp->TrainerMode)
      {
        CtlSetValue(GetCtrlPtr(form, chkOTraining), gp->TrainerMode);
        FrmAlert(AlertOnlyNew+(pp->Langu*resLF));	    
	    handled = true;
      }
      if (back)
        FrmReturnToForm(0);
      break;

     default:
      break;
  }
  return handled;
}

/****************************************************************************/
Boolean frmPieces_HandleEvent(EventPtr event)
{
FormPtr form;
Boolean handled = false;
short 	SelColor, SelPiece, x, y;
TGlobals  *pgp = GetGameGlobals();
short   otherside[3]={1,0,2};
EventType newEvent;

  switch (event->eType) {
    case frmOpenEvent:
      pgp->playingMode = false;
      form = FrmGetActiveForm();
      FrmDrawForm(form);
      pgp->DispWindowH = WinGetDrawWindow();
      if (PromotionSelection)
      {
	SelColor = otherside[hdr.opponent];
    	for (SelPiece = knight; SelPiece < king; SelPiece++)
	{
            x = (SelPiece - 1) * normSq;
            y = (normSq / 2) + TitleHeight;
    	    Copy2screen((SelPiece-1)*wdPieces, SelColor * PieceHeight,
          	  	wdPieces, PieceHeight, x, y, winPaint);
	    }
      }
      else
      {
	for (SelColor = white; SelColor <= black; SelColor++)		// Draw the pieces
    	  for (SelPiece = pawn; SelPiece <= king; SelPiece++)
	      {
            x = (SelPiece - 1) * normSq;
            y = SelColor * normSq + TitleHeight;
    	    Copy2screen((SelPiece-1)*wdPieces, SelColor * PieceHeight,
          	  	wdPieces, PieceHeight, x, y, winPaint);
	      }
      }
      handled = true;
      break;

    case penDownEvent:    // select piece
      if (prgg.isHiRes || prgg.HRrefNum)
      {
        event->screenX *= 2;
	    event->screenY *= 2;
      }
      if (PromotionSelection)
      {
	    if ((event->screenY >= ((normSq / 2) + TitleHeight)) &&
	        (event->screenY <= ((normSq / 2) + TitleHeight + PieceHeight)))
	    {
		  SelPiece = (event->screenX / normSq) + 1;
		  if ((SelPiece > pawn) && (SelPiece < king))
		  {
			PromotionSelection = SelPiece;
                        FrmReturnToForm(0);
                        pgp->DispWindowH = WinGetDrawWindow();
			ManualMove();
	  	  }
		}
      }
      else if ((event->screenY >= TitleHeight) &&
	           (event->screenY <= (TitleHeight + (2*PieceHeight))))
      {
        SelColor = (event->screenY < TitleHeight + normSq) ? black : white;
        SelPiece = (event->screenX / normSq) + 1;
	    if ((SelPiece >= pawn) && (SelPiece <= king) &&
	        ((SelColor == white) || (SelColor == black)))
        {
          color[hdr.sqSel.start] = SelColor;
          board[hdr.sqSel.start] = SelPiece; 
          InitializeStats();
          FrmReturnToForm(0);
          newEvent.eType = firstUserEvent+1;
          newEvent.data.generic.datum[0] = hdr.sqSel.start;
          EvtAddUniqueEventToQueue(&newEvent, 0, false);
          pgp->DispWindowH = WinGetDrawWindow();
          UnselectPieces();
          DrawPiece(hdr.sqSel.start);
        }  
      }
      handled = true;
      break;

    default:
      break;
    }

  return handled;
}

/****************************************************************************/
Boolean frmAbout_HandleEvent(EventPtr event) // S2
{
FormPtr form;
Boolean handled = false;
UInt32  romVersion = 0;
Char	string[41];
TGlobals  *pgp = GetGameGlobals();
EventType newEvent;

  form = FrmGetActiveForm();
  FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVersion);
      
  switch (event->eType) 
  {
    case frmOpenEvent:
      pgp->playingMode = false;
      newEvent.eType = firstUserEvent;
      EvtAddUniqueEventToQueue(&newEvent, 0, false);
      handled = true;
      break;

    case ctlEnterEvent:
      if(event->data.ctlSelect.controlID != cmdClose)
        break;
    case penDownEvent:
      if(event->screenY <= SCR_TITLE) 
        break;
      FrmReturnToForm(0);
      handled = true;
      break;

    case keyDownEvent:
	  if (event->data.keyDown.chr == vchrPageUp)
	  {
    	if(pgp->debugLevel < MAX_DBG_LEVEL)  
    	  pgp->debugLevel++;  
        handled = true;
      }
      else if (event->data.keyDown.chr == vchrPageDown)
	  {
    	if(pgp->debugLevel)  
    	  pgp->debugLevel--;  
        handled = true;
      }
      if(handled)
      {
        newEvent.eType = firstUserEvent;
        EvtAddUniqueEventToQueue(&newEvent, 0, false);
      }
      break;
      
    case firstUserEvent:
      switch (pgp->debugLevel) 
      {
        case DBG_OPENING_BOOK:
          if(pgp->Book)
            StrPrintF(string, "Book=%s.Lines=%ld.Recs=%d", 
              (pgp->BookDB?"PDB":"PRC"), pgp->OpenCnt, pgp->BookRecs);
          else
            StrPrintF(string, "No book present.");
          break;
        case DBG_15MIN_HASH_ON:
          StrPrintF(string, "Calc.15min - Hash=%dK.", (short)(pgp->ttblsz/1024L));
          break;
        case DBG_15MIN_HASH_OFF:
          StrPrintF(string, "Calc.15min - Hash off.");
          break;

        case DBG_LEVEL_RESOLUTION:
          if(romVersion >= 0x03500000)
	  {
            Coord     width = 0, height = 0;
	    UInt8     depth = 0;
            UInt16    rowBytes = 0;
	    BitmapPtr bmpPtr = WinGetBitmap(WinGetDisplayWindow());
	    
	    
	    if(bmpPtr)
	    {
	      BmpGlueGetDimensions(bmpPtr, &width, &height, &rowBytes);
	      depth = BmpGlueGetBitDepth(bmpPtr);
	    }
            StrPrintF(string, "UI %dx%dx%d [%d] %s%s%s", width, height, depth, rowBytes,
	      (pgp->isHiRes ? "HiRes":""), (pgp->HRrefNum ? "Sony":""), 
	      (pgp->HRversion==HR_VERSION_SUPPORT_FNTSIZE ? "V2":""));	  
	  }    
          break;
	          
        default:
	  {
            UInt32  width = 0, height = 0, depth = 0;
            Boolean color = 0;
	  
            if(romVersion >= 0x03503000)
              WinScreenMode(winScreenModeGet, &width, &height, &depth, &color);
            StrPrintF(string, "V%d-OS%lx-D%lx-C%d", 
              VERSION_NUM, romVersion, depth, color);
	  }    
          break;
      }
      FrmCopyLabel (form, lblSystem, string);
      FrmDrawForm(form);
      handled = true;
      break;
      
    default:
      break;
    }

  return handled;
}

/****************************************************************************/
Boolean frmMessage_HandleEvent(EventPtr event) // S2
{
FormPtr   form;
Boolean   handled = false;
Char      *string;
MemHandle old, h;
FieldPtr  fldP;
Int16	  scroll = 0;
WinDirectionType wdt = winDown;
EventType newEvent;
TGlobals  *pgp = GetGameGlobals();
UInt32    idUp, idDown;

  switch (event->eType)
  {
    case frmOpenEvent:
      form = FrmGetActiveForm();
      pgp->playingMode = false;
      h = MemHandleNew(StrLen(pgp->LibText) + 1);
      string = MemHandleLock(h);
      StrCopy(string, pgp->LibText);
      MemHandleUnlock(h);
      fldP = GetCtrlPtr(form, fldMessage);
      old = FldGetTextHandle(fldP);
      FldSetTextHandle(fldP, h);
      if(old != NULL)
        MemHandleFree(old);
      newEvent.eType = firstUserEvent;
      EvtAddUniqueEventToQueue(&newEvent, 0, false);
      FrmDrawForm(form);
      handled = true;
      break;

    case keyDownEvent:
      if(event->data.keyDown.chr == pageUpChr) 
      {
        event->data.ctlSelect.controlID = btnUp;
      } 
      else if(event->data.keyDown.chr == pageDownChr) 
      {
        event->data.ctlSelect.controlID = btnDown;
      }
    case ctlSelectEvent:
      if(event->data.ctlSelect.controlID == cmdClose)
      {
        FrmReturnToForm(0);
      }
      else if(event->data.ctlSelect.controlID == btnUp)
      {
        scroll = -1;
        wdt = winUp;
      }
      else if(event->data.ctlSelect.controlID == btnDown)
      {
        scroll = 1;
      }

      if(scroll)
      {
        form = FrmGetActiveForm();
        fldP = GetCtrlPtr(form, fldMessage);
        if(FldScrollable(fldP, wdt))
          FldScrollField(fldP, (FldGetVisibleLines(fldP) - 1), wdt);
        newEvent.eType = firstUserEvent;
        EvtAddUniqueEventToQueue(&newEvent, 0, false);
      }
      handled = true;
      break;

    case firstUserEvent:
      form = FrmGetActiveForm();
      fldP = GetCtrlPtr(form, fldMessage);
      if(!FldScrollable(fldP, wdt)) 
      {
        CtlSetEnabled(GetCtrlPtr(form, btnDown), false);
        idDown = bmpDownOff;
      }
      else
      {
        CtlSetEnabled(GetCtrlPtr(form, btnDown), true);
        idDown = bmpDown;
      }
 	  wdt = winUp;
      if(!FldScrollable(fldP, wdt)) 
      {
        CtlSetEnabled(GetCtrlPtr(form, btnUp), false);
        idUp = bmpUpOff;
      }
      else
      {
        CtlSetEnabled(GetCtrlPtr(form, btnUp), true);
        idUp = bmpUp;
      }

      if((idDown == bmpDown) || (idUp == bmpUp))
      {
        DrawButton(form, idUp, btnUp);
        DrawButton(form, idDown, btnDown);          
      }
      handled = true;
      break;
      
    default:
      break;
  }
  return handled;
}

/****************************************************************************/
Boolean frmPiecesTaken_HandleEvent(EventPtr event) // not S2
{
FormPtr form;
Boolean handled = false;
short 	SelColor, SelPiece, i, j, x, y;
short   cntPieces[2][5];
TGlobals        *pgp = GetGameGlobals();

  switch (event->eType)
  {
    case frmOpenEvent:
    {
      pgp->playingMode = false;
      form = FrmGetActiveForm();
      FrmDrawForm(form);
      pgp->DispWindowH = WinGetDrawWindow();
// init. before count
      for (SelColor = white; SelColor <= black; SelColor++)
        for (SelPiece = pawn-1; SelPiece <= queen-1; SelPiece++)
          cntPieces[SelColor][SelPiece] = 0;
// count pieces
      for (SelColor = white; SelColor <= black; SelColor++)
        for (i = 0; i <= PieceCnt[SelColor]; i++)
          if ((board[PieceList[SelColor][i]] >= pawn) &&
              (board[PieceList[SelColor][i]] <= queen))
            cntPieces[SelColor][board[PieceList[SelColor][i]]-1]++;
// draw pieces
	  i = 0;
	  for (SelPiece = pawn; SelPiece <= queen; SelPiece++)
	  {
	  j = cntPieces[white][SelPiece-1] - cntPieces[black][SelPiece-1];
		  while (j)
		  {
            if (j > 0)
	    {
	          SelColor = white;
		  j--;
	    }
            else
	    {
	          SelColor = black;
		  j++;
	    }

			if (i < 8)
			{
			  x = i * normSq;
              y = TitleHeight;
		    }
			else
			{
              x = (i - 8) * normSq;
              y = normSq + TitleHeight;
		    }
    	    Copy2screen((SelPiece-1) * wdPieces, SelColor * PieceHeight,
           	  wdPieces, PieceHeight, x, y, winPaint);

           	i++;
          }
      }
      handled = true;
      break;
	}
    case ctlEnterEvent:
      if (event->data.ctlSelect.controlID == cmdClose)
      {
          FrmReturnToForm(0);
          pgp->DispWindowH = WinGetDrawWindow();
      }
      handled = true;
      break;

    default:
      break;
    }
  return handled;
}

/****************************************************************************/
void SelectPromotionPiece(void) // not S2
{
  TOpenChessPrefs *pp = GetPrefsPtr();

  PromotionSelection = -1;
  FrmPopupForm(frmPieces+(pp->Langu*resLF));
  return;
}

/****************************************************************************/
void ManualMove(void) // not S2
{
  Boolean ok;
  UInt16  mv, nPiece;
  short otherside[3]={1,0,2};
  TOpenChessPrefs *pp = GetPrefsPtr();

  if(hdr.endgame)
    return;
    
  nPiece = PieceCnt[otherside[hdr.player]];		// The actual other side piece cnt
  ok = VerifyMove(prgg.man_mv, &mv);
  if(ok && mv != hdr.hint)
  {
    Sdepth = 0;
    ft = 0;
  }
  if(ok)
  {
    PlaySound(hdr.opponent, (nPiece != PieceCnt[otherside[hdr.player]]));
    ElapsedTime(1);
    hdr.player = hdr.computer;    // Added : computer's turn
    if(hdr.force)
    {
      hdr.computer = hdr.opponent;
      hdr.opponent = otherside[hdr.computer]; // uxdsp.c, InputCommand()
      hdr.hint = 0;
    }
    DisplayStatus();
    if(prgg.LibText)
      FrmPopupForm(frmMessage+(pp->Langu*resLF));
  }
  else if(!PromotionSelection)
    hdr.sqSel.end = NoSqSelected;
}

/****************************************************************************/
Boolean CheckBoardEdit(void) // not S2
{
  short kingCheckSq[2] = {-1, -1},
  		pawnAmt[2] = {0, 0},
  		piecesAmt[2] = {0, 0},
        sq;
  short otherside[3]={1,0,2};      

// check number of kings and no pawns on 1st or last line
  for (sq = 0; sq < 64; sq++)
  {
     if (color[sq] != neutral)
     {
// to many pieces?
       if (++piecesAmt[color[sq]] > 16)
       {
         UnselectPieces();
         hdr.sqSel.start = sq;
 	     DrawPiece(sq);
	     ShowMessage(SkipComments(17, prgg.ProgStrings));
	     return false;
       }
	   if (board[sq] == pawn)
	   {
         if (++pawnAmt[color[sq]] > 8)
         {
           UnselectPieces();
           hdr.sqSel.start = sq;
 	       DrawPiece(sq);
	       ShowMessage(SkipComments(18, prgg.ProgStrings));
	       return false;
         }
	if ((sq < 8) || (sq > 55))
	{
        UnselectPieces();
        hdr.sqSel.start = sq;
 	    DrawPiece(sq);
		ShowMessage(SkipComments(19, prgg.ProgStrings));
		return false;
	}
    }
	if (board[sq] == king)
	{
	  if (kingCheckSq[color[sq]] != -1)
	  {
            UnselectPieces();
            hdr.sqSel.start = sq;
 	    DrawPiece(sq);
 	    ShowMessage(SkipComments(20, prgg.ProgStrings));
	    return false;
	  }
	  else
	  {
	    kingCheckSq[color[sq]] = sq;
	  }
	}
     }
  }
// kings set?
  if ((kingCheckSq[white] == -1) || (kingCheckSq[black] == -1))
  {
	ShowMessage(SkipComments(21, prgg.ProgStrings));
	return false;
  }
  if(((hdr.player == white && hdr.epsquare < 32) ||
      (hdr.player == black && hdr.epsquare >= 32)) &&
      hdr.epsquare != -1)
    hdr.epsquare = -1;  
  hdr.GameCnt = -1;
  if(hdr.player == white)
    hdr.flags &= game_edit_start_black;
  else  
    hdr.flags |= game_edit_start_black;
  hdr.Game50 = 0;
  Sdepth = 0;
  InitializeStats();
  if ((distance(kingCheckSq[white],kingCheckSq[black]) == 1) ||
      (SqAtakd(PieceList[otherside[hdr.player]][0],hdr.player)))
  {
	ShowMessage(SkipComments(22, prgg.ProgStrings));
	return false;
  }
// set endgame
  hdr.endgame = 0;
  
  SetEndGame(hdr.player, 2);
  return true;
}

/****************************************************************************/
short CheckMoveListLegal(short side, short ply) // not S2
{
  short pnt,tempb,tempc,tempsf,tempst,xside,cnt = 0;
  struct leaf *node;
  short otherside[3]={1,0,2};

  xside = otherside[side];
  
  pnt = TrPnt[ply];
  while(pnt < TrPnt[ply+1])
  {
    node = &Tree[pnt++];
    MakeMove(side,node,&tempb,&tempc,&tempsf,&tempst);
    if (!SqAtakd(PieceList[side][0],xside))
      cnt++;
    else
      node->score = -30000;    
    UnmakeMove(side,node,&tempb,&tempc,&tempsf,&tempst);
  }
  return cnt;
}
  
/****************************************************************************/
void SetEndGame(short side, short ply) // not S2
{
  short pnt,tempb,tempc,tempsf,tempst,xside,cnt = 0;
  struct leaf *node;
  short otherside[3]={1,0,2};

  xside = otherside[side];
  
  if(hdr.endgame) // if set already, do not overwrite
    return;
    
  MoveList(side,ply);
  pnt = TrPnt[ply];
  while (pnt < TrPnt[ply+1])
  {
    node = &Tree[pnt++];
    MakeMove(side,node,&tempb,&tempc,&tempsf,&tempst);
    if (!SqAtakd(PieceList[side][0],xside))
      cnt++;
    UnmakeMove(side,node,&tempb,&tempc,&tempsf,&tempst);
  }
  if(cnt == 0) // no legal move left
  {
	if(hdr.GameCnt >= 0) // not after edit board
	  GameList[hdr.GameCnt].flags |= end_mask;
	  
    if (!SqAtakd(PieceList[side][0],xside))
  	  hdr.endgame = stalemate; 
    else
	  hdr.endgame = xside+1; // mate
  }
  else if(hdr.GameCnt >= 0)
  {
     if(hdr.GameCnt-hdr.Game50 >= moves_50)
     {
       hdr.endgame = rule_50_mv; 
     }
     else
     {
       short rpt = 0;
       
       repetition(&rpt);
       if(rpt >= 2)
       {
         hdr.endgame = threefold_rep; // detected only if direct repetion!!!
       }
       else if(GameList[hdr.GameCnt].flags & end_mask)
       {
         hdr.endgame = end_engine; // why not detected?
       }	 
     }  
  }
}

/****************************************************************************/
char GetAmbiguous(short f, short t, short side, short ply) // not S2
{
  short pnt,tempb,tempc,tempsf,tempst,cnt = 0;
  struct leaf *node;
  short otherside[3]={1,0,2};
  char ret_char = 0, 
       s[5];

  if((board[f] == pawn) || (board[f] == king))
    return ret_char; // not possible
  algbr(f,t);
  StrCopy(s,prgg.mvstr);
  pnt = TrPnt[ply];
  while (pnt < TrPnt[ply+1])
  {
    node = &Tree[pnt++];
    
    if((f != node->f) &&
       (board[f] == board[node->f]) &&
       (color[f] == color[node->f]) &&
       (t == node->t))
    {
      MakeMove(side,node,&tempb,&tempc,&tempsf,&tempst);
      if (!SqAtakd(PieceList[side][0],otherside[side]))
      {
        cnt++;
        algbr(node->f,node->t);
        if(s[0] == prgg.mvstr[0])
          ret_char = s[1];
        else
          ret_char = s[0];
      }
      UnmakeMove(side,node,&tempb,&tempc,&tempsf,&tempst);
    }  
  }
  if(cnt > 1)
    ret_char = multi_ambiguous;
  return ret_char;  
}

/****************************************************************************/
void DrawDbListItem(Int16 iItem, RectangleType *bounds, Char **itemsText)// S2
{
  Char        string[MAX_NAME+1];
  MemHandle   RecHand;
  TGame       *RecPtr;
  TGlobals    *pgp = GetGameGlobals();
  TOpenChessPrefs *pp = GetPrefsPtr();

  RecHand = DmQueryRecord(pgp->OpenChessDB, iItem+1);
  RecPtr = MemHandleLock(RecHand);
  if (RecPtr == NULL)
  {
    StrPrintF(string, "%d", iItem + 1);
    FrmCustomAlert(AlertError+(pp->Langu*resLF), 
      SkipComments(12, prgg.ProgStrings), string, "");
  }
  else
  {
    StrCopy(string, RecPtr->Name);
    MemHandleUnlock(RecHand);
    WinDrawChars(string, StrLen(string), bounds->topLeft.x, bounds->topLeft.y);
  }
}

/****************************************************************************/
Boolean frmDatabase_HandleEvent(EventPtr event) //S2
{
  FormPtr form = FrmGetActiveForm();
  ListPtr List = FrmGetObjectPtr(form, FrmGetObjectIndex(form, DbGameList));
  Boolean handled = false;
  TOpenChessPrefs *pp = GetPrefsPtr();
  TGlobals        *pgp = GetGameGlobals();

  switch (event->eType) {
    case frmOpenEvent:
      pgp->playingMode = false;
	  LstSetDrawFunction(List, DrawDbListItem);
	  LstSetListChoices(List, NULL, DmNumRecords(pgp->OpenChessDB) - 1);
	  LstSetSelection(List, pp->DbGameIndex - 1);
	  FrmDrawForm(form);
      handled = true;
      break;

    case ctlEnterEvent:
      if (event->data.ctlSelect.controlID == btnOK)
      {
        Int16 iSelected = LstGetSelection(List);

        if ((iSelected != noListSelection) && (iSelected + 1 < DmNumRecords(pgp->OpenChessDB)))
	{
          SaveGameToDB(pp->DbGameIndex);
	  ReadGameFromDB(pp->DbGameIndex = 1 + iSelected);
	}
        handled = true;
      }
      else
        if (event->data.ctlSelect.controlID == btnCancel)
	  handled = true;

      if (handled)
        BackToMain();
      break;

    default:
      break;
    }

  return handled;
}

/****************************************************************************/
void BackToMain(void) // S2
{
  TOpenChessPrefs *pp = GetPrefsPtr();
	
  if (pp->fullMode)
    FrmGotoForm(frmMainFull);
  else
    FrmGotoForm(frmMain);
}

/****************************************************************************/
void EndOfOpening(Boolean ok) // S2
{
  Int16   button;
  TOpenChessPrefs *pp = GetPrefsPtr();
  TGlobals        *pgp = GetGameGlobals();
  TGame           *gp = GetGamePtr();

  if (!gp->TrainerMode)
    return;

  if(ok)
  {
    if(!pp->fullMode)
    {
      EventType event;

      ShowMessage(SkipComments(30, pgp->ProgStrings));

      do
      {
        EvtGetEvent(&event, SysTicksPerSecond() / 4);
        if(SysHandleEvent(&event))
	      break;
      }
      while((event.eType != penDownEvent) && (event.eType != appStopEvent));
    }
    button = FrmAlert(OpeningEndOK+(pp->Langu*resLF));
    button++;
  }
  else
  {
    if(!pp->fullMode)
    {
      EventType event;

      ShowMessage(SkipComments(23, pgp->ProgStrings));

      do
      {
        EvtGetEvent(&event, SysTicksPerSecond() / 4);
        if(SysHandleEvent(&event))
	      break;
      }
      while((event.eType != penDownEvent) && (event.eType != appStopEvent));
    }
    button = FrmAlert(OpeningEndWrong+(pp->Langu*resLF));
  }

  switch (button)
  {
    case 0: // Try Again
        pgp->Book = pgp->BookPtr;
        Back();
        GiveHint();
        break;
    case 1: // New test
        pgp->Book = pgp->BookPtr;
        while(gp->GameCnt >= gp->opponent)
          Undo();
        if(gp->GameCnt == 0)
        {
          Undo();
          gp->player = gp->computer; //white in this case
        }
        pgp->mvstr[0] = '\0';
        gp->sqSel.start = gp->sqSel.end = NoSqSelected;
        UpdateDisplay(0,0,1,0);
        DisplayStatus();
        InitializeStats();
	    break;
    case 2: // Play Now
        gp->TrainerMode = false;
        Back();
        gp->sqSel.start = GameList[hdr.GameCnt+1].gmove>>8;
        gp->sqSel.end = GameList[hdr.GameCnt+1].gmove & 0xFF;
        algbr(gp->sqSel.start,gp->sqSel.end);
        StrCopy(pgp->man_mv,pgp->mvstr);  
        ManualMove();
        break;
  }
}

/****************************************************************************/
void SetFullMode(Boolean fm) //S2
{
  TOpenChessPrefs *pp = GetPrefsPtr();
  TGlobals  *pgp = GetGameGlobals();

  if(fm)
  {
    pgp->szSq = fullSq;
	pgp->offBoard = fullOffset;
  }
  else
  {
    pgp->szSq = normSq;
    pgp->offBoard = normOffset;
  }
  pp->fullMode = fm;
}

/****************************************************************************/
void FreeLibMem(void) //S2
{
  UInt16 i = 0;
  TGlobals  *pgp = GetGameGlobals();
  
  if(pgp->nLib)
  {
    while(i < pgp->nLib)
      MemPtrFree(pgp->LibNames[i++]);
    MemPtrFree(pgp->LibNames);
    pgp->nLib     = 0;
    pgp->LibNames = NULL;
  }
}

/****************************************************************************/
void FreeBookMem(void) //S2
{
  TGlobals         *pgp = GetGameGlobals();
  UInt16           i,
                   **List;
 
  if(pgp->BookPtr)
    MemPtrFree(pgp->BookPtr);
  if(pgp->BookDB)
  {
    if(pgp->BookDataList)
    {
      List = pgp->BookDataList;
      for(i = 0; i < pgp->BookRecs; i++)
        MemPtrUnlock(*List++); 
      MemPtrFree(pgp->BookDataList);
    }
    DmCloseDatabase(pgp->BookDB);
  }
  else
  {
    if(pgp->BookData)
      MemPtrUnlock(pgp->BookData);
    if(pgp->BookHdl)
      DmReleaseResource(pgp->BookHdl);
  }
  
  pgp->LibText = NULL;
  pgp->LibRootText = NULL;
  pgp->BookHdl = NULL;
  pgp->BookData = NULL;
  pgp->BookDataList = NULL;
  pgp->Book = NULL;
  pgp->BookPtr = NULL;
  pgp->BookDB = 0;
  pgp->BookRecs = 0;
  pgp->OpenCnt = 0L;  
}

/****************************************************************************/
void AllocHashMem(void) // not S2
{
  if(prgg.debugLevel == DBG_15MIN_HASH_OFF)
  {
    FreeHashMem();
    return;
  }
  else if(ttable)
  {
    return;
  }
  
  FreeBookMem();

  for(prgg.ttblsz = start_ttblsz; prgg.ttblsz >= min_ttblsz; prgg.ttblsz /= 2L)  
  {
    ttable = (struct hashentry *)MemGluePtrNew((UInt32)prgg.ttblsz*(unsigned long)sizeof(struct hashentry));
    if(ttable)
      break;
  }
  if(!ttable)
    prgg.ttblsz = 0L;
}

/****************************************************************************/
void FreeHashMem(void) // not S2
{
  if(ttable)
    MemPtrFree(ttable);
  ttable = NULL;
  prgg.ttblsz = 0L;
}

/****************************************************************************/
char *SkipComments(Int16 i, char *txtPtr) // S2
{
  UInt16 skips;

  while(i)
  {
	if(*txtPtr)
	{
	  while(*txtPtr++);
	  i--;
	}
	else
	{
	  txtPtr++;
	  skips = (UInt8)*txtPtr++;
	  i = i - skips;
	  if(i < 0)
	    return NULL;
	}
  }
  return txtPtr;
}

/****************************************************************************/
void GetProgStrings(void) // S2
{
  MemHandle hdlProgStrings;
  TOpenChessPrefs *pp = GetPrefsPtr();
  TGlobals  *pgp = GetGameGlobals();
  Char      *txtPtr,
            *destPtr;

  if(pgp->ProgStrings)
    MemPtrFree(pgp->ProgStrings);
    
  hdlProgStrings = DmGetResource('tSTR', strProgStrings+(pp->Langu*resLF));
  txtPtr = MemHandleLock(hdlProgStrings);
  
  destPtr = pgp->ProgStrings = MemPtrNew(StrLen(txtPtr));
  CRASHIF(!destPtr, "No memory.");
  if(pgp->ProgStrings)
  {
    while(*txtPtr)
    {
	  if(*txtPtr == '\n')  
	    *destPtr++ = '\0';
	  else
	    *destPtr++ = *txtPtr;  
	  txtPtr++;  
    }
  }  
  MemHandleUnlock(hdlProgStrings);
  DmReleaseResource(hdlProgStrings);
}

/****************************************************************************/
void CheckRules(short sq) // not S2
{
  short pc = 0,
        cl = 0,
	eps = 0,
	*ptrMoved = NULL;
  TOpenChessPrefs *pp = GetPrefsPtr();

  switch (sq)
  {
    case 0:
      pc = rook;
      cl = white;
      ptrMoved = &(hdr.lrookmoved[cl]);
      break;
    case 4:
      pc = king;
      cl = white;
      ptrMoved = &(hdr.kingmoved[cl]);
      break;
    case 7:
      pc = rook;
      cl = white;
      ptrMoved = &(hdr.hrookmoved[cl]);
      break;
    case 56:
      pc = rook;
      cl = black;
      ptrMoved = &(hdr.lrookmoved[cl]);
      break;
    case 60:
      pc = king;
      cl = black;
      ptrMoved = &(hdr.kingmoved[cl]);
      break;
    case 63:
      pc = rook;
      cl = black;
      ptrMoved = &(hdr.hrookmoved[cl]);
      break;
    case 24:
    case 25:
    case 26:
    case 27:
    case 28:
    case 29:
    case 30:
    case 31:
     if((board[sq] == pawn) && (color[sq] == white))
        eps = sq-8;
      break;
    case 32:
    case 33:
    case 34:
    case 35:
    case 36:
    case 37:
    case 38:
    case 39:
      if((board[sq] == pawn) && (color[sq] == black))
        eps = sq+8;
      break;
      
    default:
      return; // exit
  }

  if((board[sq] == pc) && (color[sq] == cl) && (ptrMoved != NULL))
  {
    if(FrmAlert(AlertPieceMoved+(pp->Langu*resLF)) == 0)
      *ptrMoved = 100;
    else
      *ptrMoved = 0;
  }
  else if(eps)
  {
  	if(FrmAlert(AlertEPPawn+(pp->Langu*resLF)) == 0)
      hdr.epsquare = eps;
  }
}

/****************************************************************************/
TOpenChessPrefs *GetPrefsPtr(void)
{
  return &prefs;
}

/****************************************************************************/
TGame *GetGamePtr(void)
{
  return &hdr;
}

/****************************************************************************/
TGlobals *GetGameGlobals(void)
{
  return &prgg;
}

/****************************************************************************/
Char *MoveString(short gcnt, unsigned short flags)
{
  TGlobals *pgp = GetGameGlobals();
  short f, t, i = 0, comp;
  Boolean long_not;
  
if(gcnt >= 0)
{
  if(hdr.flags & game_edit_start_black)
    comp = 1;
  else
    comp = 0;  

  if(flags & 0x0001)
    long_not = true;
  else
    long_not = false;
    
  if(flags & 0x0002)
  {
	StrPrintF(prgg.last_mv_string, "%d.%s ", (gcnt+comp+2)/2, ((gcnt+comp+2)%2 ? ".." : ""));  
	i = StrLen(prgg.last_mv_string);
  }
    
  f = GameList[gcnt].gmove>>8;
  t = GameList[gcnt].gmove & 0xFF;
  
  if(GameList[gcnt].flags & cstlmask)
  {
      prgg.last_mv_string[i++] = 'O';
      prgg.last_mv_string[i++] = '-';
      prgg.last_mv_string[i++] = 'O';
      if(t < f)
      {
        prgg.last_mv_string[i++] = '-';
        prgg.last_mv_string[i++] = 'O';
      }
  }
  else
  {
    algbr(f, t);  
    if(GameList[gcnt].move_piece != pawn)
    {
      if(flags & 0x0004)
        prgg.last_mv_string[i++] = pxx[GameList[gcnt].move_piece];
      else	
        prgg.last_mv_string[i++] = *(pgp->ProgStrings + GameList[gcnt].move_piece);
    }  
    else if((GameList[gcnt].flags & capture) && !long_not)
      prgg.last_mv_string[i++] = pgp->mvstr[0];
    if((GameList[gcnt].ambiguous == multi_ambiguous) || long_not)
    {
      prgg.last_mv_string[i++] = pgp->mvstr[0];
      prgg.last_mv_string[i++] = pgp->mvstr[1];
    }  
    else if(GameList[gcnt].ambiguous)
      prgg.last_mv_string[i++] = GameList[gcnt].ambiguous;
    if(GameList[gcnt].flags & capture)
      prgg.last_mv_string[i++] = 'x';
    else if(long_not)
      prgg.last_mv_string[i++] = '-';
    prgg.last_mv_string[i++] = pgp->mvstr[2];
    prgg.last_mv_string[i++] = pgp->mvstr[3];
    if(GameList[gcnt].flags & promote)
    {
      prgg.last_mv_string[i++] = '=';
      prgg.last_mv_string[i++] = *(pgp->ProgStrings + GameList[gcnt].prom_piece);
    }
  }
  if((GameList[gcnt].flags & end_mask) && (GameList[gcnt].flags & check))
    prgg.last_mv_string[i++] = '#';
  else if(GameList[gcnt].flags & check)
    prgg.last_mv_string[i++] = '+';
  prgg.last_mv_string[i] = '\0';
}  
return prgg.last_mv_string;  
}

/****************************************************************************/
void PGNExport(void) // S2
{
  DateTimeType dtt;	
  TGame *gp = GetGamePtr();
  TGlobals *pgp = GetGameGlobals();
  TOpenChessPrefs *pp = GetPrefsPtr();
	
  if(gp->flags & game_flag_edit)
    FrmCustomAlert(AlertError+(pp->Langu*resLF), 
      SkipComments(10, pgp->ProgStrings), "", "");
  else
  {  
    DmOpenRef dbMemo;
    MemHandle hdlMemo;
    MemPtr    pMemo;
    Char      memoBuffer[MEMO_BUFFER],
              result[10];
    short     memoSize = 0,
              i,cnt;
    UInt16    dbSize, dbIndex = dmMaxRecordIndex;
    Char      strDay[] = "00";
    Char      strMonth[] = "00";
    
    dbMemo = DmOpenDatabaseByTypeCreator('DATA', 'memo', dmModeReadWrite);
    if(!dbMemo)
    {
      FrmCustomAlert(AlertError+(pp->Langu*resLF), 
        SkipComments(10, pgp->ProgStrings), "DB.", "");
      return;
    }
    dbSize = MEMO_START;
    hdlMemo = DmNewRecord(dbMemo, &dbIndex, dbSize);
    pMemo = MemHandleLock(hdlMemo);
    if(pMemo)
    {
	  short w, b;
	  
      if(gp->bothsides)
      {
	    w = b = 1;
      }
      else if (gp->force)
      {
	    w = b = 0;
      }
      else if (gp->opponent == white)
      {
	    w = 0; 
	    b = 1;
      }
      else
      {
	    w = 1; 
	    b = 0;
      }
   	  StrPrintF(memoBuffer, "[Event \"%s\"]\n", *gp->Name ? gp->Name : "OpenChess");
      DmStrCopy(pMemo, memoSize, memoBuffer);
      memoSize += StrLen(memoBuffer);
      
      TimSecondsToDateTime(TimGetSeconds(), &dtt);
      if(dtt.day > 10)
        StrPrintF(strDay, "%d", dtt.day);
      else
        strDay[1] += dtt.day;
      if(dtt.month > 10)
        StrPrintF(strMonth, "%d", dtt.month);
      else
        strMonth[1] += dtt.month;
   	  StrPrintF(memoBuffer, "[Date \"%d.%s.%s\"]\n", dtt.year, strMonth, strDay);
      DmStrCopy(pMemo, memoSize, memoBuffer);
      memoSize += StrLen(memoBuffer);
      
   	  StrPrintF(memoBuffer, "[White \"%s\"]\n", 
   	    SkipComments(27+w, pgp->ProgStrings));
      DmStrCopy(pMemo, memoSize, memoBuffer);
      memoSize += StrLen(memoBuffer);
   	  StrPrintF(memoBuffer, "[Black \"%s\"]\n", 
   	    SkipComments(27+b, pgp->ProgStrings));
      DmStrCopy(pMemo, memoSize, memoBuffer);
      memoSize += StrLen(memoBuffer);
      
      switch (gp->endgame)
      {
        case 1:
          StrCopy(result, "1-0");
          break;
        case 2:
          StrCopy(result, "0-1");
          break;
        case 3:
        case 4:
        case 5:
          StrCopy(result, "1/2-1/2");
          break;
      	case 0:
      	default:
          StrCopy(result, "*");
          break;
      }
      StrPrintF(memoBuffer, "[Result \"%s\"]\n", result);
      DmStrCopy(pMemo, memoSize, memoBuffer);
      memoSize += StrLen(memoBuffer);
      i = cnt = 0;
      while(cnt <= gp->GameCnt)
      {
		if(memoSize + MEMO_PLUS > dbSize)
        {
		  MemHandleUnlock(hdlMemo);
		  dbSize += MEMO_PLUS;
		  hdlMemo = DmResizeRecord(dbMemo, dbIndex, dbSize);
		  if(!hdlMemo)
		  {
            FrmCustomAlert(AlertError+(pp->Langu*resLF), 
              SkipComments(10, pgp->ProgStrings), "Resize.", "");
            return;
		  }
          pMemo = MemHandleLock(hdlMemo);
        }
        StrPrintF(memoBuffer, "%s ", MoveString(cnt++, (i++%2 ? 0x04 : 0x06)));
        DmStrCopy(pMemo, memoSize, memoBuffer);
        memoSize += StrLen(memoBuffer);
      }
      StrPrintF(memoBuffer, "%s\n", result);
      DmStrCopy(pMemo, memoSize, memoBuffer);
    }
    else
    {
      FrmCustomAlert(AlertError+(pp->Langu*resLF), 
        SkipComments(10, pgp->ProgStrings), "Lock.", "");
      return;
    }
    MemHandleUnlock(hdlMemo);
    DmReleaseRecord(dbMemo, dbIndex, true);
    DmCloseDatabase(dbMemo);
    FrmAlert(AlertExport+(pp->Langu*resLF));
  }
}

/****************************************************************************/
void OpenUI(void) // S2
{
  TGlobals  *pgp = GetGameGlobals();
  Err       err;
  UInt32    depth, attr;
  SonySysFtrSysInfoP sonySysFtrSysInfoP;
  Coord	      width, height;
  MemHandle   resH;
  BitmapPtr   resP;
  Boolean     color;
  UInt32      bmpPieces,
              wScreen,
              hScreen;
  RectangleType rect;
  
  /* Check for OS5 hi-res display.  First - is Window Mgr new enough? */
  err = FtrGet(sysFtrCreator, sysFtrNumWinVersion, &pgp->WinVersion);
  if(!err && (pgp->WinVersion >= 4)) 
  {
    /*
     * Window Manager version is greater than 4 - it will at least
     * understand the question about high-density
     */
    err = WinScreenGetAttribute(winScreenDensity, &attr);
    pgp->isHiRes = (!err && (attr == kDensityDouble));
  }
	
  /* Check for preOS5 Sony HiRes */  
  if(!pgp->isHiRes && (pgp->WinVersion < 5))
  {
    err = FtrGet(sonySysFtrCreator, sonySysFtrNumSysInfoP, (UInt32*)&sonySysFtrSysInfoP);
    if(!err && (sonySysFtrSysInfoP->libr & sonySysFtrSysInfoLibrHR))
    {
	  err = SysLibFind(sonySysLibNameHR, &pgp->HRrefNum);
	  if(err == sysErrLibNotFound) 
        err = SysLibLoad( 'libr', sonySysFileCHRLib, &pgp->HRrefNum);
      if(!err)  
      {
        HRGetAPIVersion(pgp->HRrefNum, &pgp->HRversion);
        err = HROpen(pgp->HRrefNum);
        if(!err) 
	{
	  if(pgp->HRversion < HR_VERSION_SUPPORT_FNTSIZE) 
	  {
            wScreen = hrWidth,
            hScreen = hrHeight;
	    
            depth = 0x010;
            HRWinScreenMode(pgp->HRrefNum, winScreenModeSet, &wScreen, &hScreen, &depth, NULL);
	  }  
        }
      }
    }  
  }
  WinScreenMode(winScreenModeGet, NULL, NULL, &depth, &color);
  if(!color)
  {
    depth = 4;
    WinScreenMode(winScreenModeSet , NULL, NULL, &depth, NULL);
    WinScreenMode(winScreenModeGet, NULL, NULL, &depth, &color);
    if(depth < 4)
    {
      depth = 2;
      WinScreenMode(winScreenModeSet , NULL, NULL, &depth, NULL);
      WinScreenMode(winScreenModeGet, NULL, NULL, &depth, &color);
    }
  }
  else
  {
    depth = 0x010;
    WinScreenMode(winScreenModeSet , NULL, NULL, &depth, NULL);
    WinScreenMode(winScreenModeGet, NULL, NULL, &depth, &color);
  }
  
// select BMP fitting for resolution/color depth
  if(pgp->isHiRes || pgp->HRrefNum)
  {
	pgp->density = 2;  
    bmpPieces = bmpPiecesLarge;
  }
  else
  {
	pgp->density = 1;  
    bmpPieces = bmpPiecesSmall;
  }

  // Load pieces bitmap
  resH = DmGetResource(bitmapRsc, bmpPieces);

  // Create offscreen window, and load pieces bitmap
  resP = MemHandleLock(resH);

  BmpGlueGetDimensions(resP, &width, &height, NULL);

  rect.topLeft.x = 
  rect.topLeft.y = 0;
  rect.extent.x = width;
  rect.extent.y = height;
  
  if(pgp->isHiRes)
  {
    /* Do the trick described in Palm doco - create double density bitmap
     * and map to an off-screen window - this stops pixel-doubling when copying
     */
    pgp->v3bmp = BmpCreateBitmapV3(resP, kDensityDouble, BmpGetBits(resP), NULL);
    pgp->OffscreenWinH = WinCreateBitmapWindow((BitmapPtr)pgp->v3bmp, &err);
    pgp->DispWindowH = WinGetDrawWindow();
  }
  else if(pgp->HRrefNum)
  {
    /* Sony pre-OS5 HiRes  */
    pgp->OffscreenWinH = HRWinCreateOffscreenWindow(pgp->HRrefNum, width, height, screenFormat, &err);
    if(pgp->OffscreenWinH)
    {
      pgp->DispWindowH = WinSetDrawWindow(pgp->OffscreenWinH);
      HRWinEraseRectangle(prgg.HRrefNum, &rect, 0);
      HRWinDrawBitmap(pgp->HRrefNum, resP, 0, 0);
    }
  }
  else if(pgp->WinVersion >= 5)
  {
    BitmapPtr bmpLast,
              bmpPtr = resP;

    while(BmpGetBitDepth(bmpPtr) < ((UInt8)depth))
    {
      bmpLast = bmpPtr;
      bmpPtr = BmpGetNextBitmap(bmpLast);
      if(!bmpPtr)
      {
        bmpPtr = bmpLast;
	    break;
      }
    }
    pgp->v3bmp = BmpCreateBitmapV3(bmpPtr, kDensityLow, BmpGetBits(bmpPtr), NULL);
    pgp->OffscreenWinH = WinCreateBitmapWindow((BitmapPtr)pgp->v3bmp, &err);
    pgp->DispWindowH = WinGetDrawWindow();
  }
  else
  {
    /* Lo-res (160x160) display */
    pgp->OffscreenWinH = WinCreateOffscreenWindow(width, height, screenFormat, &err);
    if(pgp->OffscreenWinH)
    {
      pgp->DispWindowH = WinSetDrawWindow(pgp->OffscreenWinH);
      WinEraseRectangle(&rect, 0); 
      WinDrawBitmap(resP, 0, 0);
    }
  }

  MemPtrUnlock(resP);
  DmReleaseResource(resH);
  WinSetDrawWindow(pgp->DispWindowH);    // Restore the current draw window
}

/****************************************************************************/
void SetLibText(void) // S2
{
  TGame *gp = GetGamePtr();
  TGlobals *pgp = GetGameGlobals();
  short j;
  unsigned short *mp;
  struct BookEntry *p;
  UInt32 i = 0;

  pgp->LibText = NULL;
  
  if(!gp->TrainerMode)
    return;
    
  if(gp->GameCnt < 0)  
  {
    pgp->LibText = pgp->LibRootText;
    return;
  }
 
  p = pgp->Book;
  while (i++ < pgp->OpenCnt)
  {
    mp = p->mv;
    for(j = 0; j <= gp->GameCnt; j++)
       if(BOOK_MOVE(j) != *(mp++))
         break;

    if(j > gp->GameCnt)
    {
	  char *textPtr;

      if(p->comments)
	  {
        textPtr = SkipComments(gp->GameCnt + 1, p->comments);
	    if(textPtr)
	    {
	      if(*textPtr)
	      {
            pgp->LibText = textPtr;
            break;
          }
        }
      }
    }
    p++;
  }
}

/****************************************************************************/
void DrawBmp(UInt32 bmpID, Coord x, Coord y) // S2
{
  TGlobals       *pgp = GetGameGlobals();
  MemHandle      resH;
  BitmapPtr      resP;
  BitmapPtrV3    bmpV3 = NULL;

  resH = DmGetResource(bitmapRsc, bmpID+((pgp->HRrefNum || pgp->isHiRes)?1:0));
  resP = MemHandleLock(resH);
  
  if(pgp->HRrefNum)
  {
    HRWinDrawBitmap(pgp->HRrefNum, resP, x, y);
  }
  else
  {
    if(pgp->isHiRes)
    {
      bmpV3 = BmpCreateBitmapV3(resP, kDensityDouble, BmpGetBits(resP), NULL);
      WinSetCoordinateSystem(kCoordinatesNative);
      WinDrawBitmap((BitmapPtr)bmpV3, x, y);
      WinSetCoordinateSystem(kCoordinatesStandard);
    }
    else
    {
      WinDrawBitmap(resP, x, y);
    }
  }
  MemPtrUnlock(resP);
  DmReleaseResource(resH);
  if(bmpV3)
    BmpDelete((BitmapType *)bmpV3);
}

/****************************************************************************/
void DrawButton(FormPtr form, UInt32 bmpID, UInt32 btnID) // S2
{
  TGlobals       *pgp = GetGameGlobals();
  RectangleType  rect;
  MemHandle      resH;
  BitmapPtr      resP;
  Coord	         width, 
                 height;
  UInt16	 offX = 0,
                 offY = 0;

  resH = DmGetResource(bitmapRsc, bmpID+((pgp->HRrefNum || pgp->isHiRes)?1:0));
  resP = MemHandleLock(resH);
  BmpGlueGetDimensions(resP, &width, &height, NULL);
  MemPtrUnlock(resP);
  DmReleaseResource(resH);
  
  
  FrmGetObjectBounds(form, FrmGetObjectIndex(form, btnID), &rect);
  rect.topLeft.x *= pgp->density; 
  rect.topLeft.y *= pgp->density; 
  rect.extent.x *= pgp->density; 
  rect.extent.y *= pgp->density; 

  if(pgp->HRrefNum)
  {
    HRWinEraseRectangle(pgp->HRrefNum, &rect, 0);
  }
  else if(pgp->isHiRes)
  {
    WinSetCoordinateSystem(kCoordinatesNative);
    WinEraseRectangle(&rect, 0);
    WinSetCoordinateSystem(kCoordinatesStandard);
  }
  else
  {
    WinEraseRectangle(&rect, 0);
  }
  
  if(rect.extent.x > width)
    offX = (rect.extent.x - width)/2;
  if(rect.extent.y > height)
    offY = (rect.extent.y - height)/2;
  DrawBmp(bmpID, rect.topLeft.x+offX, rect.topLeft.y+offY);
}

/*
{Char str40[40] = "";
  StrPrintF(str40, "%d", xxx);
  FrmCustomAlert(AlertError, "", str40, "");}    
*/



