/*
 * rc2Main.c
 *
 * main file for Radio Control
 *
 * Copyright 2003 Chris Hobbs
 *
 * See http://www.clanhobbs.org/treo for more Treo
 * specific software.
 * 
 * This software is released under the terms of the 
 * General Public License, a copy of which you should
 * have received along with this source file. Should
 * you need a copy of this license, please visit:
 * http://www.fsf.org/licenses/gpl.txt
 * 
 * The author makes no warranties, express or implied,
 * concerning the use of this code. Data loss, memory
 * leaks, small fires, and the harming of baby seals
 * are all likely outcomes from using this code - 
 * proceed at your own risk.
 *
 * Questions or comments should be directed to 
 * the author at treo_coder@clanhobbs.org.
 *
 */
 
 /*
 
 Version history...
 
 1.0
 ---
 5/28/03
 Initial release
 
 1.0.1
 -----
 6/2/03
 Removed debugging windows.
 
 1.1
 ---
 7/06/03
 Added code to reset alarm after soft reset.
 
 1.1-650
 -------
 1/01/05
 Modified to support 650
 
 
 */
 
#include <68K/Hs.h>
/* #include <PhoneGlobals.h>
#include <HsExt.h> */

#include "RadioControl_res.h"

// defines for different things

#define appFileCreator			'TrRC'
#define appName					"Radio Control"
#define appVersionNum			0x01
#define appPrefID				0x00
#define appPrefVersionNum		0x01

// Override the defines for Treo300

#define phnLibName300 "PhoneLib.prc"
#define	phnLibDbType300	'libr'
#define	phnLibDbCreator300	'HsCL'

typedef struct {
	UInt8		resetState;
	TimeType	onTime;
	TimeType	offTime;
} RCPreferenceType;


#define RESET_ON	1
#define RESET_OFF	2
#define ALARM_ON	1
#define	ALARM_OFF	2
#define NO_TIME	100

// globals

UInt8 gResetState;
TimeType gOnTime, gOffTime;
UInt16			PhoneLibRefNum;	// Phone library reference number

static void * GetObjectPtr(UInt16 objectID)
{
    FormType * frmP;

    frmP = FrmGetActiveForm();
    return FrmGetObjectPtr(frmP, FrmGetObjectIndex(frmP, objectID));
}


static Err StartApplication (void)
{
	// Load app preferences if they exist...
	RCPreferenceType prefs;
	UInt16	prefsSize;
	
	prefsSize = sizeof(RCPreferenceType);
	if (PrefGetAppPreferences(appFileCreator, appPrefID, &prefs, 
			&prefsSize, true) != noPreferenceFound)
	{
		gResetState = prefs.resetState;
		gOnTime	= prefs.onTime;
		gOffTime = prefs.offTime;
		
		
		
		
	} else { // no prefs exist yet...
		gResetState = RESET_OFF;
		gOnTime.hours = gOnTime.minutes = gOffTime.hours = gOffTime.minutes = NO_TIME;
	}
		
	FrmGotoForm(MainForm);
	return 0;
}

static void StopApplication (void)
{
	
	RCPreferenceType prefs;
	UInt16	cardNo;
	LocalID	dbID;
	DmSearchStateType	searchInfo;
	UInt32 alarmTime, nowTime, ref;
	DateTimeType	nowStruct;
	Err error;
	/*
	char Time[64];
	char Date[64];
	char TRef[64];
	*/
	
	// Write out prefs
	prefs.resetState = gResetState;
	prefs.onTime = gOnTime;
	prefs.offTime = gOffTime;
	
	PrefSetAppPreferences(appFileCreator, appPrefID, appPrefVersionNum, 
						&prefs, sizeof(prefs), true);
	
	// set next alarm (or cancel)
	
	DmGetNextDatabaseByTypeCreator (true, &searchInfo, sysFileTApplication, 
					appFileCreator, true, &cardNo, &dbID);
	if (gOnTime.hours == NO_TIME && gOffTime.hours == NO_TIME)
	{
		AlmSetAlarm(cardNo, dbID, 0, 0, true); // clears alarm
	} else {
		Int16 onTimeDiff, offTimeDiff;
		
		nowTime = TimGetSeconds();
		TimSecondsToDateTime(nowTime, &nowStruct);
		// figure next alarm...
		onTimeDiff = ((gOnTime.hours - nowStruct.hour) * 100);
		if (onTimeDiff >= 0)
			onTimeDiff += (gOnTime.minutes - nowStruct.minute);
		else
			onTimeDiff -= (gOnTime.minutes - nowStruct.minute);
			
		if (gOnTime.hours == NO_TIME)
			onTimeDiff = -1; 
			
		offTimeDiff = ((gOffTime.hours - nowStruct.hour) * 100);
		if (offTimeDiff >= 0)
			offTimeDiff += (gOffTime.minutes - nowStruct.minute);
		else
			offTimeDiff -= (gOffTime.minutes - nowStruct.minute);
			
		if (gOffTime.hours == NO_TIME)
			offTimeDiff = -1;
		
		if (onTimeDiff >= 0 && offTimeDiff >= 0)
		{
			if (onTimeDiff <= offTimeDiff)
			{
				nowStruct.hour = gOnTime.hours;
				nowStruct.minute = gOnTime.minutes;
				ref = ALARM_ON;
			} else {
				nowStruct.hour = gOffTime.hours;
				nowStruct.minute = gOffTime.minutes;
				ref = ALARM_OFF;
			}
		} else if (onTimeDiff < 0 && offTimeDiff < 0)
		{
			if (onTimeDiff <= offTimeDiff)
			{
				TimAdjust(&nowStruct, 86400); // add one day
				nowStruct.hour = gOnTime.hours;
				nowStruct.minute = gOnTime.minutes;				
				ref = ALARM_ON;
			} else {
				TimAdjust(&nowStruct, 86400); // add one day
				nowStruct.hour = gOffTime.hours;
				nowStruct.minute = gOffTime.minutes;
				ref = ALARM_OFF;
			}
		} else if (onTimeDiff >= 0 && offTimeDiff < 0)
		{
			nowStruct.hour = gOnTime.hours;
			nowStruct.minute = gOnTime.minutes;
			ref = ALARM_ON;
		} else if (offTimeDiff >= 0 && onTimeDiff < 0)
		{
			nowStruct.hour = gOffTime.hours;
			nowStruct.minute = gOffTime.minutes;
			ref = ALARM_OFF;
		} else
		{
			// should never get here...
		}
		alarmTime = TimDateTimeToSeconds(&nowStruct);
		error = AlmSetAlarm(cardNo, dbID, ref, alarmTime, true);
		/*
		if(!error)
		{
			TimeToAscii(nowStruct.hour, nowStruct.minute, tfColonAMPM, Time);
			DateToAscii(nowStruct.month, nowStruct.day, nowStruct.year, dfMDYWithSlashes, Date);
			StrIToA(TRef, ref);
			FrmCustomAlert(AlarmSetAlert, Time, Date, TRef);
		}
		*/
	}
}

static void AlarmTriggered (SysAlarmTriggeredParamType *params)
{
	RCPreferenceType prefs;
	UInt16	prefsSize;
	Err error = 0;
	UInt32 alarmTime, nowTime, ref;
	DateTimeType	nowStruct;
	UInt16	cardNo;
	LocalID	dbID;
	DmSearchStateType	searchInfo;
//	char Time[64];
		
	prefsSize = sizeof(RCPreferenceType);
	if (PrefGetAppPreferences(appFileCreator, appPrefID, &prefs, 
			&prefsSize, true) != noPreferenceFound)
	{
		nowTime = TimGetSeconds();
		TimSecondsToDateTime(nowTime, &nowStruct);
		if ((params->ref == ALARM_ON) || (prefs.onTime.hours == NO_TIME))
		{
			ref = ALARM_OFF;
			if ((nowStruct.hour < prefs.offTime.hours) ||
					(nowStruct.hour == prefs.offTime.hours &&
					 nowStruct.minute < prefs.offTime.minutes))
			{
				nowStruct.hour = prefs.offTime.hours;
				nowStruct.minute = prefs.offTime.minutes;
			} else {
				TimAdjust(&nowStruct, 86400); // add one day
				nowStruct.hour = prefs.offTime.hours;
				nowStruct.minute = prefs.offTime.minutes;
			}
		} else {
			ref = ALARM_ON;
			if ((nowStruct.hour < prefs.onTime.hours) ||
					(nowStruct.hour == prefs.onTime.hours &&
					 nowStruct.minute < prefs.onTime.minutes))
			{
				nowStruct.hour = prefs.onTime.hours;
				nowStruct.minute = prefs.onTime.minutes;
			} else {
				TimAdjust(&nowStruct, 86400); // add one day
				nowStruct.hour = prefs.onTime.hours;
				nowStruct.minute = prefs.onTime.minutes;
			}
		}
		DmGetNextDatabaseByTypeCreator (true, &searchInfo, sysFileTApplication, 
					appFileCreator, true, &cardNo, &dbID);
		alarmTime = TimDateTimeToSeconds(&nowStruct);
		error = AlmSetAlarm(cardNo, dbID, ref, alarmTime, true);
		/*
		if (!error)
		{
			TimeToAscii(nowStruct.hour, nowStruct.minute, tfColonAMPM, Time);
			FrmCustomAlert(AlarmSetAlert, Time, NULL, NULL);
		}
		*/
	}
}

static void HandleAlarm (SysDisplayAlarmParamType * params)
{
	Err error = 0;
	UInt16	PhoneLibRefNum;	// Phone library reference number
	UInt32 libType, libCreator, phoneType;
	char *phoneLibName[40];
	
	HsAttrGet(hsAttrPhoneType, 0, &phoneType);
	
	if (phoneType == hsAttrPhoneTypeCDMA) // Treo 300
	{
		
		*phoneLibName = phnLibCDMAName;
		libType = phnLibDbType300;
		libCreator = phnLibCDMADbCreator;
	} else {
		*phoneLibName = phnLibGSMName;
		libType = phnLibDbType;
		libCreator = phnLibGSMDbCreator;
	}
	
	
	error = SysLibFind (*phoneLibName, &PhoneLibRefNum);
	if (error)
	{
		error = SysLibLoad (libType, libCreator, &PhoneLibRefNum);
		if (error)
		{
			FrmCustomAlert (PowerErrorAlert, "SysLibLoad failed", NULL, NULL);
		}
	}
	if (params->ref == ALARM_ON)
	{
		error = PhnLibSetModulePower(PhoneLibRefNum, true);
		if (error)
		{
			FrmCustomAlert (PowerErrorAlert, "Phone poweron failed", NULL, NULL);
		}
	} else {
		error = PhnLibSetModulePower(PhoneLibRefNum, false);
		if (error)
		{
			FrmCustomAlert (PowerErrorAlert, "Phone poweroff failed", NULL, NULL);
		}
	}
}

static Boolean AboutMenuHandleEvent (UInt16 menuID)
{
	Boolean handled = false;
	FormType	*form;
		
	form = FrmGetActiveForm();
	switch (menuID) {
		case AboutAboutRadioControl:
			FrmAlert(AboutAlert);
			handled = true;
			break;
		case AboutInstructions:
			FrmAlert(InstructionsAlert);
			handled = true;
			break;
		default:
			break;
	}
	
	return handled;
}

static Boolean MainFormHandleEvent (EventPtr event)
{
	Boolean handled = false;
	
	switch (event->eType) {
		case frmOpenEvent:
		{
			FormType *form = FrmGetActiveForm();
			FrmDrawForm(form);
			{ 
				char Time[64];
				ControlType *ctl = GetObjectPtr(MainOnTimeSelTrigger);
				if (gOnTime.hours == NO_TIME)
				{
					CtlSetLabel(ctl, "Never");
				} else {
					TimeToAscii (gOnTime.hours, gOnTime.minutes, tfColonAMPM, Time);
					CtlSetLabel(ctl, Time);
				}
				ctl = GetObjectPtr(MainOffTimeSelTrigger);
				if (gOffTime.hours == NO_TIME)
				{
					CtlSetLabel(ctl, "Never");
				} else {
					TimeToAscii (gOffTime.hours, gOffTime.minutes, tfColonAMPM, Time);
					CtlSetLabel(ctl, Time);
				}
				if (gResetState == RESET_ON)
				{
					FormType *form = FrmGetActiveForm();
					FrmSetControlGroupSelection (form, MainGroupID, MainResetOnPushButton);
				} else {
					FormType *form = FrmGetActiveForm();
					FrmSetControlGroupSelection (form, MainGroupID, MainResetOffPushButton);
				}
				
			}
			handled = true;
		}
		
			break;
		case  ctlSelectEvent:
			switch (event->data.ctlSelect.controlID) {
				case MainResetOnPushButton:
					gResetState = RESET_ON;
					{
						FormType *form = FrmGetActiveForm();
						FrmSetControlGroupSelection (form, MainGroupID, MainResetOnPushButton);
					}
					handled = true;
					break;
				case MainResetOffPushButton:
					gResetState = RESET_OFF;
					{
						FormType *form = FrmGetActiveForm();
						FrmSetControlGroupSelection (form, MainGroupID, MainResetOffPushButton);
					}
					handled = true;
					break;
				case MainOnTimeSelTrigger:
				{
					Int16 hour = 0;
					Int16 min = 0;
					
					if (gOnTime.hours != NO_TIME)
					{
						hour = gOnTime.hours;
						min = gOnTime.minutes;
					}
					
					if (SelectOneTime (&hour, &min, "Radio On Time"))
					{
						char Time[64];
						ControlType *ctl = GetObjectPtr(MainOnTimeSelTrigger);
						gOnTime.hours = hour;
						gOnTime.minutes = min;
						TimeToAscii (hour, min, tfColonAMPM, Time);
						CtlSetLabel(ctl, Time);						
					}
					handled = true;					
				}
					break;
				case MainOffTimeSelTrigger:
				{
					Int16 hour = 0;
					Int16 min = 0;
					
					if (gOffTime.hours != NO_TIME)
					{
						hour = gOffTime.hours;
						min = gOffTime.minutes;
					}
					
					if (SelectOneTime (&hour, &min, "Radio Off Time"))
					{
						char Time[64];
						ControlType *ctl = GetObjectPtr(MainOffTimeSelTrigger);
						gOffTime.hours = hour;
						gOffTime.minutes = min;
						TimeToAscii (hour, min, tfColonAMPM, Time);
						CtlSetLabel(ctl, Time);
					}
					handled = true;					
				}
					break;
				// Add cleartime option here
				case MainClearTimesButton:
					gOnTime.hours = gOnTime.minutes = gOffTime.hours = gOffTime.minutes = NO_TIME;
					{
						ControlType *ctl = GetObjectPtr(MainOnTimeSelTrigger);
						CtlSetLabel(ctl, "Never");
						ctl = GetObjectPtr(MainOffTimeSelTrigger);
						CtlSetLabel(ctl, "Never");
					}
					break;
				default:
					break;
			}
			break;
		case menuEvent:
			handled = AboutMenuHandleEvent(event->data.menu.itemID);
			break;
		default:
			break;
	}
	return handled;
}

static Err SystemReset (void)
{
	RCPreferenceType prefs;
	UInt16	prefsSize;
	Err error = 0;
	UInt16	PhoneLibRefNum;	// Phone library reference number
	UInt32 libType, libCreator, phoneType;
	char *phoneLibName[40];
	UInt16	cardNo;
	LocalID	dbID;
	DmSearchStateType	searchInfo;
	UInt32 alarmTime, nowTime, ref;
	DateTimeType	nowStruct;
	TimeType gOnTime, gOffTime; // globals aren't available, redefine here
	/*
	char Time[64];
	char Date[64];
	char TRef[64];
	*/
	
	HsAttrGet(hsAttrPhoneType, 0, &phoneType);
	
	if (phoneType == hsAttrPhoneTypeCDMA) // Treo 300
	{
		
		*phoneLibName = phnLibCDMAName;
		libType = phnLibDbType300;
		libCreator = phnLibCDMADbCreator;
	} else {
		*phoneLibName = phnLibGSMName;
		libType = phnLibDbType;
		libCreator = phnLibGSMDbCreator;
	}
	
	prefsSize = sizeof(RCPreferenceType);
	if (PrefGetAppPreferences(appFileCreator, appPrefID, &prefs, 
			&prefsSize, true) != noPreferenceFound)
	{
				
		if (prefs.resetState == RESET_ON) // deal with wireless mode
		{
			error = SysLibFind (*phoneLibName, &PhoneLibRefNum);
	  		if (error)
			{
				error = SysLibLoad (libType, libCreator, &PhoneLibRefNum);
				if (error)
				{
					FrmCustomAlert (PowerErrorAlert, "SysLibLoad failed", NULL, NULL);
					return (error);
				}
			}
			error = PhnLibSetModulePower(PhoneLibRefNum, true);
			if (error)
			{
				FrmCustomAlert (PowerErrorAlert, "PhnLibSetModulePower failed", NULL, NULL);
				return (error);
			}
		}
		
		// reset alarm (alarms are cleared after reset)
		
		gOnTime = prefs.onTime;
		gOffTime = prefs.offTime;
		
		DmGetNextDatabaseByTypeCreator (true, &searchInfo, sysFileTApplication, 
					appFileCreator, true, &cardNo, &dbID);
		
		if (gOnTime.hours == NO_TIME && gOffTime.hours == NO_TIME)
		{
			AlmSetAlarm(cardNo, dbID, 0, 0, true); // clears alarm
		} else {
			Int16 onTimeDiff, offTimeDiff;
			
			nowTime = TimGetSeconds();
			TimSecondsToDateTime(nowTime, &nowStruct);
			// figure next alarm...
			onTimeDiff = ((gOnTime.hours - nowStruct.hour) * 100);
			if (onTimeDiff >= 0)
				onTimeDiff += (gOnTime.minutes - nowStruct.minute);
			else
				onTimeDiff -= (gOnTime.minutes - nowStruct.minute);
				
			if (gOnTime.hours == NO_TIME)
				onTimeDiff = -1; 
				
			offTimeDiff = ((gOffTime.hours - nowStruct.hour) * 100);
			if (offTimeDiff >= 0)
				offTimeDiff += (gOffTime.minutes - nowStruct.minute);
			else
				offTimeDiff -= (gOffTime.minutes - nowStruct.minute);
				
			if (gOffTime.hours == NO_TIME)
				offTimeDiff = -1;
			
			if (onTimeDiff >= 0 && offTimeDiff >= 0)
			{
				if (onTimeDiff <= offTimeDiff)
				{
					nowStruct.hour = gOnTime.hours;
					nowStruct.minute = gOnTime.minutes;
					ref = ALARM_ON;
				} else {
					nowStruct.hour = gOffTime.hours;
					nowStruct.minute = gOffTime.minutes;
					ref = ALARM_OFF;
				}
			} else if (onTimeDiff < 0 && offTimeDiff < 0)
			{
				if (onTimeDiff <= offTimeDiff)
				{
					TimAdjust(&nowStruct, 86400); // add one day
					nowStruct.hour = gOnTime.hours;
					nowStruct.minute = gOnTime.minutes;				
					ref = ALARM_ON;
				} else {
					TimAdjust(&nowStruct, 86400); // add one day
					nowStruct.hour = gOffTime.hours;
					nowStruct.minute = gOffTime.minutes;
					ref = ALARM_OFF;
				}
			} else if (onTimeDiff >= 0 && offTimeDiff < 0)
			{
				nowStruct.hour = gOnTime.hours;
				nowStruct.minute = gOnTime.minutes;
				ref = ALARM_ON;
			} else if (offTimeDiff >= 0 && onTimeDiff < 0)
			{
				nowStruct.hour = gOffTime.hours;
				nowStruct.minute = gOffTime.minutes;
				ref = ALARM_OFF;
			} else
			{
				// should never get here...
			}
			alarmTime = TimDateTimeToSeconds(&nowStruct);
			error = AlmSetAlarm(cardNo, dbID, ref, alarmTime, true);
			/*
			if(!error)
			{
				TimeToAscii(nowStruct.hour, nowStruct.minute, tfColonAMPM, Time);
				DateToAscii(nowStruct.month, nowStruct.day, nowStruct.year, dfMDYWithSlashes, Date);
				StrIToA(TRef, ref);
				FrmCustomAlert(AlarmSetAlert, Time, Date, TRef);
			}
			*/
		}
		
	} else { // no prefs exist yet, do nothing...
		
	}
	
	return error;
}
	

static Boolean ApplicationHandleEvent (EventPtr event)
{
	FormType *form;
	UInt16 formID;
	Boolean handled = false;
	
	if (event->eType == frmLoadEvent) {
		formID = event->data.frmLoad.formID;
		form = FrmInitForm(formID);
		FrmSetActiveForm(form);
		
		switch (formID) {
			case MainForm:
				FrmSetEventHandler(form, MainFormHandleEvent);
				break;
			default:
				break;
		}
		handled = true;
	}
	return handled;
}

static void EventLoop (void)
{
	EventType event;
	UInt16 error;
	
	do
	{
		EvtGetEvent(&event, evtWaitForever);
		if (! SysHandleEvent(&event))
			if (! MenuHandleEvent (0, &event, &error))
				if (! ApplicationHandleEvent(&event))
					FrmDispatchEvent(&event);
	} while (event.eType != appStopEvent);
}

UInt32 PilotMain (UInt16 launchCode, MemPtr cmdPBP,
					UInt16 launchFlags)
{
	Err err;
	
	switch (launchCode) {
		case sysAppLaunchCmdNormalLaunch:
			if ((err = StartApplication()) == 0)
			{
				EventLoop();
				StopApplication();
			}
			break;
		case sysAppLaunchCmdSystemReset:
			SystemReset();
			break;
		case sysAppLaunchCmdAlarmTriggered:
			AlarmTriggered((SysAlarmTriggeredParamType *) cmdPBP);
			break;
		case sysAppLaunchCmdDisplayAlarm:
			HandleAlarm((SysDisplayAlarmParamType *) cmdPBP);
			break;
		default:
			break;
	}
	return err;
}