/*
 * ZLPlugin.h
 *
 *
 * Version: 1.0B
 *
 * Copyright (c) 2004 Guangzhou Zhangzhe technology Co., Ltd..
 * All rights reserved.
 *
 * Release History:
 *   1.0B - Sep 14, 2004
 *
 */
#ifndef ZLPLUGIN_H_
#define ZLPLUGIN_H_

#include <PalmOS.h>

#define ZLPluginAppType 	 'ZLPN'	//	Type ID of the Plugin app
#define ZLPluginCodeRscNum   1		//	Code Resource Number of the Plugin app which contains the implementation of
									//	the Plugin

#define	ZLPluginRscFirstID	30000	//	All IDs of plugin's resources must be not less than 30000,
									//	to avoid conflicting with ZL's resources
									//	Before using plugin's resources, it's recommended to open the plugin database
									//	first, to make sure to get the correct resource. Or directly get the resource
									//	from the plugin database.
//	Recommended Resources
#define	ZLPluginIconNameRscID		31000
#define	ZLPluginLargeIconRscID		31000
#define	ZLPluginSmallIconRscID		31001
#define	ZLPluginVersionStringRscID	31000


//	Error codes
//
#define	ZLPluginErrNotSupportedCmd	appErrorClass

//	Plugin events
//
#define	lastZLPluginEvent	0x7FFF
#define	ZLDragAndDropEvent	((eventsEnum)(lastZLPluginEvent	-	5))	//	Drag & Drop event. Details in ZLPluginLaunchCmdEventType
#define	ZLDeactivateEvent	((eventsEnum)(lastZLPluginEvent	-	6))

//	Plugin launch code
//
#define	sysAppLaunchCmdZLPluginLoad	(sysAppLaunchCmdCustomBase)

//	Parameter block for Drag & Drop event
//
typedef	struct
{
	UInt32	numItems;
	
	struct
	{
		Boolean	internal;	//	Indicates the item is in the internal of the handheld or on the expansion card
		UInt8	padding[3];
		
		union
		{
			struct	
			{
				UInt16	cardNo;
				UInt8	padding[2];
				LocalID	dbID;
			}	internal;	//	the item is in the internal of the handheld
			
			struct	
			{
				UInt16	volRefNum;
				UInt8	padding[2];
				char	*path;
			}	external;	//	the item is on the expansion card
		}	location;
		
	}	item[];
	
} ZLDragAndDropType;

//	Defines the position constant
//
typedef enum
{
	ZLPluginPosTab,		//	the plugin is on Plugin Tab
	ZLPluginPosPanel,	//	the plugin is on Plugin Panel
	ZLPluginPosButton	//	the plugin is on Top/Bottom Info Panel
} ZLPluginPositionType;

typedef enum	//	only available when the plugin is on Plugin Panel
{
	ZLPluginPanelTop,		//	Plugin panel is on the top of the screen
	ZLPluginPanelLeft,		//	Plugin panel is on the left side of the screen
	ZLPluginPanelBottom,	//	Plugin panel is on the bottom of the screen
	ZLPluginPanelRight		//	Plugin panel is on the right ride of the screen
} ZLPluginPanelType;

typedef enum	//	only available when the plugin is on Top/Bottom Info Panel
{
	ZLPluginButtonTop,		//	the plugin is on Top Info Panel
	ZLPluginButtonBottom	//	the plugin is on Bottom Info Panel
} ZLPluginButtonType;

//	Interfaces implemented by the Plugin app
typedef enum
{
	ZLPluginFuncPtrQuery,		//	Interface for querying the necessary information of the plugin app.
								//	This function is expected to be called by ZL at any time, even the plugin is not
								//	opened and initialized, or the plugin has already been closed.
								//	Should not access plugin's global inside this function.
								
	ZLPluginFuncPtrOpen,		//	Initializes the resources of the plugin app, like the globals used by the plugin
	ZLPluginFuncPtrClose,		//	Releases the resources of the plugin app
	
	ZLPluginFuncPtrHandleEvent,	//	Default event handler of the plugin app
	ZLPluginFuncPtrRefresh,		//	The draw function of the plugin. Do custom drawing here.
	
	ZLPluginFuncPtrAbout,		//	Implements the About screen
	ZLPluginFuncPtrPreferences,	//	Preferences for the plugin app

	ZLPluginNumFuncPtrs
} ZLPluginFuncPtrType;

//	Functions provided by ZLauncher
//
typedef enum
{
	ZLFuncPtrReopen,				//	After calling this function, ZLauncher will close the plugin and reopen it.
	
	ZLFuncPtrBeginDraw,				//	Call this function before doing custom drawing.
									//	If lockScreen is set to TRUE, the display will be frozen while doing
									//	the custom drawing.
									//	Because this function copies the screen, using it is a relatively
									//	expensive operation.
	ZLFuncPtrEndDraw,				//	Call this function after doing custom drawing.
									//	If lockScreen was set to TRUE when ZLBeginDraw() was called,
									//	the display will be unlocked and updated after calling this function.
									
	ZLFuncPtrPrepareHandlingEvent,	//	Call this function before handling pen events,
									//	refer to ZLPluginSmaple for more details.
									
	ZLFuncPtrSetHint,				//	Set the hints of the plugin.
	ZLFuncPtrShowHint,				//	Show the hints of the plugin.
	ZLFuncPtrHideHint,				//	Hide the hints of the plugin.

//	Each plugin instance can has its own preferences, a plugin instance's preferences are transparent to other
//	plugin instance, even they are same plugin app.
	ZLFuncPtrPrefsReadValue,			//	Read a specific value from the plugin's preferences
	ZLFuncPtrPrefsReadVariableValue,	//	Read a specific variable-length value from the plugin's preferences
	ZLFuncPtrPrefsWriteValue,			//	Write a specific value to the plugin's preferences
	ZLFuncPtrPrefsWriteVariableValue,	//	Write a specific variable-length value to the plugin's preferences

	ZLFuncPtrRefreshBackground,		//	Refresh the background of the plugin
	
	ZLFuncPtrDrawChars,				//	Works like WinDrawTruncChars, but ignores the background color.
									//	Plugin can use this function to draw transparent characters
									//	If you don't want to truncate the characters, set "width" to zero, 
	ZLFuncPtrDrawBitmap,			//	Works like WinDrawBitmap. This function always draws zoomed-bitmap of a
									//	bitmap family. On all hi-res(including pre-OS5 clies)/hi-density handhelds,
									//	the hi-density version	of the bitmap will be drawn if a hi-density bitmap
									//	is available.
	
	ZLFuncNumPtrs
} ZLFuncPtrType;

//	Defines mask constant for the keys used for Prefs Feature
//
#define KeyKind1ByteMask        0x1000	//	Mask for 1-byte key
#define KeyKind2ByteMask        0x2000	//	Mask for 2-byte key
#define KeyKind4ByteMask        0x3000	//	Mask for 4-byte key
#define KeyKindVariableByteMask 0xF000	//	Mask for variable-byte key

#define KeyKindMask             0xF000
#define KeyIDMask               0x0FFF


//	Event data block for ZLPluginFuncPtrHandleEvent
typedef struct
{
	EventType	*eventP;		//	Pointer to the event which might be handled by the plugin
	Boolean		handled;		//	TRUE if the event is handled by the plugin
	UInt8		paddings[3];
} ZLPluginLaunchCmdEventType;



typedef struct
{
	void					**pZLFuncPtrs;			//	Pointers points to the functions provided by ZLauncher
	UInt32					pluginMgrVerNum;		//	The version number of the current ZL Plugin Manager
	
	ZLPluginPositionType	position;				//	Position of the plugin: Plugin Tab, Plugin Panel or Plugin Button.
	ZLPluginPanelType		panelType;				//	Available only when position is ZLPluginPosPanel
	ZLPluginButtonType		buttonType;				//	Available only when position is ZLPluginPosButton
	Boolean					lefty;					//	Indicates that user requires lefty support
	
	Int16					ZLPluginPanelWidth;		//	The width of Plugin Panel
	Int16					ZLPluginPanelHeight;	//	The height of Plugin Panel
	
	RectangleType			bounds;		//	The bounds of the plugin
	RectangleType			clipBounds;	//	The clip bounds for the plugin in ZLauncher

	DmOpenRef				dbP;		//	The pointer to the plugin datatbase
	void					*globalP;	//	Pointer to the global block of the plugin.
										//	The global block should be allocated and
										//	initialized by the plugin when receiving ZLPluginLaunchCmdOpen,
										//	and released when receiving ZLPluginLaunchCmdClose
	void					*dataP;		//	If cmdType is ZLPluginLaunchCmdHandleEvent, dataP points to	the ZLPluginLaunchCmdEventType structure.
										//	If cmdType is ZLPluginLaunchCmdQuery, dataP points to ZLPluginLaunchCmdQueryType structure
} ZLPluginLaunchCmdBlockType;

#define ZLPluginMgrVerNum 0x01000000L	//	The version of Plugin Manager. Current version is v.1.0.0

//	Data block for ZLPluginFuncPtrQuery
typedef struct
{
	UInt32	supportedZLPluginMgrVerNum;	//	The version number of the supported ZL Plugin Manager,
										//	Should be ZLPluginMgrVerNum
										
	Int32	updateInterval;		//	The event update time-out(unit: Tick), -1 means wait forever
	
	Coord	minWidth;	//	The minimum width  expected by the plugin.
						//	If the available room is smaller than the minimum requirement,
						//	the plugin will not be initialized
	Coord	minHeight;	//	The minimum height expected by the plugin.
						//	If the available room is smaller than the minimum requirement,
						//	the plugin will not be initialized
	Coord	maxWidth;	//	The maximum width expected by the plugin.
						//	If the available room is larger than the maximum requirement,
						//	only the maximum required room will be available for the plugin.
	Coord	maxHeight;	//	The maximum height expected by the plugin.
						//	If the available room is larger than the maximum requirement,
						//	only the maximum required room will be available for the plugin.

//	Note:
//		If the plugin app is on Tab, minWidth, minHeight, maxWidth will be ignored
//		If the plugin app is on Panel:
//			Top/Bottom:	minHeight and maxHeight will be ignored
//			Left/Right: minWidth and maxWidth will be ignored
//		If the plugin app is on Top/Bottom bar, minWidth, minHeight and maxHeight will be ignored
//
//		If minWidth < maxWidth, the plugin app is horizontal resize-able
//		If minHeight < maxHeight, the plugin app is vertical resize-able
//							
} ZLPluginLaunchCmdQueryType;



typedef Err	(*ZLPluginFuncPtr)	(ZLPluginLaunchCmdBlockType * blockP);
//	Note:
//		blockP is only available during the call of the function.
//		When the called function ends, blockP becomes invalid, so you
//		should never use blockP out side the called function.
//		However, blockP->globalP and blockP->dbP are always available
//		until the plugin is closed(when ZLPluginFuncPtrClose() is called).
//

//	Defines macros for function-call to the ZL functions
//

typedef void	ZLReopenF					(ZLPluginLaunchCmdBlockType * blockP);

typedef void	ZLBeginDrawF				(ZLPluginLaunchCmdBlockType * blockP,	Boolean	lockScreen);
typedef void	ZLEndDrawF					(ZLPluginLaunchCmdBlockType * blockP);

typedef void	ZLPrepareHandlingEventF		(ZLPluginLaunchCmdBlockType * blockP);

typedef void	ZLSetHintF					(ZLPluginLaunchCmdBlockType * blockP, Char * hints);
typedef void	ZLShowHintF					(ZLPluginLaunchCmdBlockType * blockP, Coord x, Coord y);
typedef void	ZLHideHintF					(ZLPluginLaunchCmdBlockType * blockP);

//Return a copy of a plugins preferences value.
//
//Parameters:
//	id				->	ID number of the preferences value to retrieve.
//
//	defaultValue	->	Default value
//
//Returns:	
//	Returns the value retrieved from the preferences, or
//	returns defaultValue if the preferences wasnt found.
typedef UInt32	ZLPrefsReadValueF			(ZLPluginLaunchCmdBlockType * blockP, UInt16 id, UInt32 defaultValue);


//Return a copy of a plugins preferences variable-length value.
//
//Parameters:
//	id		->	ID number of the value to retrieve.
//
//	valueP	->	Pointer to a buffer to hold the preferences.
//
//	sizeP	<->	Pointer to the size of the value buffer passed in.
//				(Note that the pointer and the value to which it points must be initialized
//				 before you call ZLPrefsReadVariableValue.)
//				Upon return, contains the number of bytes actually written or the number of
//				bytes needed for the prefs structure.
//			
//				To determine the required size for the value structure, set sizeP to 0 and pass NULL for valueP.
//				Upon return, the sizeP parameter contains the required size. Never set valueP to NULL without
//				also setting sizeP to 0.
//
//				Always compare the value returned in this parameter with
//				the value you passed in. If the two values differ, you need to
//				resize the prefs structure and call this function again.
//
//Returns:	
//	Returns the number of bytes retrieved from the preferences, or
//	returns the constant noPreferenceFound if the preferences wasnt found.
typedef UInt16	ZLPrefsReadVariableValueF	(ZLPluginLaunchCmdBlockType * blockP, UInt16 id, void * valueP, UInt16 * sizeP);

//Writes a value to a plugins preferences.
//
//Parameters:
//	id		->	ID number of the preferences value to write.
//
//	value	->	Default value
//
//Returns:	
//	Returns the number of bytes written to the preferences, or
//	returns 0 if the preferences value wasnt written correctly.
typedef UInt16	ZLPrefsWriteValueF			(ZLPluginLaunchCmdBlockType * blockP, UInt16 id, UInt32 value);

//Writes a variable-length value to a plugins preferences.
//
//Parameters:
//	id		->	ID number of the preferences value to write.
//
//	valueP	->	Pointer to a buffer that holds the current value of the preferences.
//
//	size	->	Size of the buffer passed
//
//Returns:	
//	Returns the number of bytes written to the preferences, or
//	returns 0 if the preferences value wasnt written correctly.
typedef UInt16	ZLPrefsWriteVariableValueF	(ZLPluginLaunchCmdBlockType * blockP, UInt16 id, void * valueP, UInt16 size);

typedef void	ZLRefreshBackgroundF		(ZLPluginLaunchCmdBlockType * blockP);
typedef void	ZLDrawCharsF				(ZLPluginLaunchCmdBlockType * blockP, Char * chars, Coord x, Coord y, Coord width);
typedef void	ZLDrawBitmapF				(ZLPluginLaunchCmdBlockType * blockP, BitmapType * pBitmap, Coord x, Coord y);



#define	ZLReopen(blockP)					((ZLReopenF *)(blockP->pZLFuncPtrs[ZLFuncPtrReopen])) (blockP)

#define	ZLBeginDraw(blockP, lockScreen)		((ZLBeginDrawF *)(blockP->pZLFuncPtrs[ZLFuncPtrBeginDraw])) ((blockP), (lockScreen))
#define	ZLEndDraw(blockP)					((ZLEndDrawF *)(blockP->pZLFuncPtrs[ZLFuncPtrEndDraw])) (blockP)

#define	ZLPrepareHandlingEvent(blockP)		((ZLPrepareHandlingEventF *)(blockP->pZLFuncPtrs[ZLFuncPtrPrepareHandlingEvent])) (blockP)

#define	ZLSetHint(blockP, hints)			((ZLSetHintF *)(blockP->pZLFuncPtrs[ZLFuncPtrSetHint])) ((blockP), (hints))
#define	ZLShowHint(blockP, x, y)			((ZLShowHintF *)(blockP->pZLFuncPtrs[ZLFuncPtrShowHint])) ((blockP), (x), (y))
#define	ZLHideHint(blockP)					((ZLHideHintF *)(blockP->pZLFuncPtrs[ZLFuncPtrHideHint])) ((blockP))

#define	ZLPrefsReadValue(blockP, id, defaultValue)			((ZLPrefsReadValueF *)(blockP->pZLFuncPtrs[ZLFuncPtrPrefsReadValue])) ((blockP), (id), (defaultValue))
#define	ZLPrefsReadVariableValue(blockP, id, valueP, sizeP)	((ZLPrefsReadVariableValueF *)(blockP->pZLFuncPtrs[ZLFuncPtrPrefsReadVariableValue])) ((blockP), (id), (valueP), (sizeP))
#define	ZLPrefsWriteValue(blockP, id, value)				((ZLPrefsWriteValueF *)(blockP->pZLFuncPtrs[ZLFuncPtrPrefsWriteValue])) ((blockP), (id), (value))
#define	ZLPrefsWriteVariableValue(blockP, id, valueP, size)	((ZLPrefsWriteVariableValueF *)(blockP->pZLFuncPtrs[ZLFuncPtrPrefsWriteVariableValue])) ((blockP), (id), (valueP), (size))

#define	ZLRefreshBackground(blockP)				((ZLRefreshBackgroundF *)(blockP->pZLFuncPtrs[ZLFuncPtrRefreshBackground])) (blockP)
#define	ZLDrawChars(blockP, chars, x, y, width)	((ZLDrawCharsF *)(blockP->pZLFuncPtrs[ZLFuncPtrDrawChars])) ((blockP), (chars), (x), (y), (width))
#define	ZLDrawBitmap(blockP, pBitmap, x, y)		((ZLDrawBitmapF *)(blockP->pZLFuncPtrs[ZLFuncPtrDrawBitmap])) ((blockP), (pBitmap), (x), (y))

#endif /* ZLPLUGIN_H_ */
