/*
 * ZLPluginSample.c
 *
 * main file for ZLPluginSample
 *
 * Version: 1.0B
 *
 * Copyright (c) 2004 Guangzhou Zhangzhe technology Co., Ltd..
 * All rights reserved.
 */
 
#include <PalmOS.h>

#include "ZLPlugin.h"

#include "ZLPluginSample.h"
#include "ZLPluginSample_Rsc.h"

/*********************************************************************
 * Global variables
 *********************************************************************/

typedef struct
{
	Coord			x;
	Coord			y;
	Coord			xOffset;
	Coord			yOffset;
	Char			iconName[dmDBNameLength];
	RGBColorType	color;
	BitmapType		*bitmapP;
	LocalID			dbID;
	UInt16			cardNo;
	Int8			xDirection;
	Int8			yDirection;
	Boolean			boolAllowAnimation;
} ZLPluginSampleGlobalType;

/*********************************************************************
 * Internal Constants
 *********************************************************************/

typedef enum
{
	KeyIDCreator = 0x0000,
	KeyIDColorRedBits,
	KeyIDColorGreenBits,
	KeyIDColorBlueBits

}	PrefsKeyIDs;

#define PrefsKeyIDCreator			((UInt16)(KeyKind4ByteMask | KeyIDCreator))
#define PrefsKeyIDColorRedBits		((UInt16)(KeyKind1ByteMask | KeyIDColorRedBits))
#define PrefsKeyIDColorGreenBits	((UInt16)(KeyKind1ByteMask | KeyIDColorGreenBits))
#define PrefsKeyIDColorBlueBits		((UInt16)(KeyKind1ByteMask | KeyIDColorBlueBits))

#define	WIDTH	10
#define	HEIGHT	10

/*********************************************************************
 * Internal Functions
 *********************************************************************/
static Err ZLPluginAbout(ZLPluginLaunchCmdBlockType * blockP);
static Err ZLPluginClose(ZLPluginLaunchCmdBlockType * blockP);
static Err ZLPluginHandleEvent(ZLPluginLaunchCmdBlockType * blockP);
static Err ZLPluginOpen(ZLPluginLaunchCmdBlockType * blockP);
static Err ZLPluginPreferences(ZLPluginLaunchCmdBlockType * blockP);
static Err ZLPluginQuery(ZLPluginLaunchCmdBlockType * blockP);
static Err ZLPluginRefresh(ZLPluginLaunchCmdBlockType * blockP);

/*
 * FUNCTION: PrvGetAppIcon
 *
 * DESCRIPTION:
 *
 * This routine gets the icon name of the app and store it in iconName.
 *
 */

static void PrvGetAppIcon(ZLPluginLaunchCmdBlockType * blockP)
{
	DmOpenRef					dbP;
	ZLPluginSampleGlobalType	*globalP;
	MemHandle					handle;
	
	globalP = (ZLPluginSampleGlobalType	*)(blockP->globalP);

	if(globalP->bitmapP	!=	0)
	{
		MemPtrFree(globalP->bitmapP);
		globalP->bitmapP	=	0;
	}
	//	Get the icon of the app.
	//
	if(globalP->dbID	!=	0)
	{
		dbP	=	DmOpenDatabase(globalP->cardNo,	globalP->dbID,	dmModeReadOnly);
		if(dbP	!=	0)
		{
			switch(blockP->position)
			{
				case ZLPluginPosTab:	//	the plugin is on the Plugin Tab
					handle	=	DmGet1Resource(iconType, 1000);
					break;
					
				case ZLPluginPosPanel:	//	the plugin is on the Plugin Panel
					handle	=	DmGet1Resource(iconType, 1001);
					break;
				
				default:
					handle	=	0;
					break;
			}				
				
			if(handle	!=	0)
			{
				UInt32	size;
				
				size	=	MemHandleSize(handle);
				
				globalP->bitmapP	=	(BitmapType *)MemPtrNew(size);
				if(globalP->bitmapP	!=	0)
				{
					MemMove(globalP->bitmapP,	MemHandleLock(handle),	size);
				}
				MemHandleUnlock(handle);
			}

			DmCloseDatabase(dbP);
		}
	}

}

/*
 * FUNCTION: PrvGetAppIconName
 *
 * DESCRIPTION:
 *
 * This routine gets the icon name of the app and store it in iconName.
 *
 */

static void PrvGetAppIconName(ZLPluginLaunchCmdBlockType * blockP)
{
	ZLPluginSampleGlobalType	*globalP;
	MemHandle					handle;
	
	globalP = (ZLPluginSampleGlobalType	*)(blockP->globalP);

	//	Get the icon name of the app
	//
	globalP->iconName[0]	=	NULL;	
	if(globalP->dbID	!=	0)
	{
		DmOpenRef	dbP;
		
		dbP	=	DmOpenDatabase(globalP->cardNo,	globalP->dbID,	dmModeReadOnly);
		if(dbP	!=	0)
		{
			handle	=	DmGet1Resource(ainRsc, ainID);
			if(handle	!=	0)
			{
				StrCopy(globalP->iconName,	(Char	*)MemHandleLock(handle));
				MemHandleUnlock(handle);
			}
			DmCloseDatabase(dbP);
		}
		if(globalP->iconName[0]	==	NULL)	//	The app dose not contain a icon name, display "- None -" instead
		{
			DmDatabaseInfo(globalP->cardNo,	globalP->dbID, globalP->iconName, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
		}
	}
	
	if(globalP->iconName[0]	==	NULL)	//	The app dose not contain a icon name, display "- None -" instead
	{
		UInt16	index;
		
		index	=	DmFindResource(blockP->dbP,	strRsc, NoneString,	0);
		if(index	!=	-1)
		{
			handle	=	DmGetResourceIndex(blockP->dbP,	index);
			if(handle	!=	0)
			{
				StrCopy(globalP->iconName,	(Char	*)MemHandleLock(handle));
				MemHandleUnlock(handle);
			}
		}
	}
	
}

/*
 * FUNCTION: PrvSetHint
 *
 * DESCRIPTION:
 *
 * This routine sets the icon name of the app as the hint.
 *
 */
static void PrvSetHint(ZLPluginLaunchCmdBlockType * blockP)
{
	ZLPluginSampleGlobalType	*globalP;
	
	globalP = (ZLPluginSampleGlobalType	*)(blockP->globalP);

	ZLSetHint(blockP, globalP->iconName);
}					


static Err ZLPluginAbout(ZLPluginLaunchCmdBlockType * blockP)
{
	FormType * frmP;

	/* Display the About Box. */
	frmP = FrmInitForm (AboutForm);
	FrmDoDialog (frmP);                    
	FrmDeleteForm (frmP);

	return errNone;
}

static Err ZLPluginClose(ZLPluginLaunchCmdBlockType * blockP)
{
	ZLPluginSampleGlobalType	*globalP;
	UInt32						creator;

	globalP = (ZLPluginSampleGlobalType	*)(blockP->globalP);

	if(globalP	!=	0)
	{
		DmDatabaseInfo (globalP->cardNo, globalP->dbID,	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, &creator);

		ZLPrefsWriteValue(blockP, PrefsKeyIDCreator, creator);

		ZLPrefsWriteValue(blockP, PrefsKeyIDColorRedBits, globalP->color.r);
		ZLPrefsWriteValue(blockP, PrefsKeyIDColorGreenBits, globalP->color.g);
		ZLPrefsWriteValue(blockP, PrefsKeyIDColorBlueBits, globalP->color.b);
		
		if(globalP->bitmapP	!=	0)
		{
			MemPtrFree(globalP->bitmapP);
			globalP->bitmapP	=	0;
		}

		MemPtrFree(globalP);
	}

	blockP->globalP	=	0;

	return errNone;
}


static Err ZLPluginHandleEvent(ZLPluginLaunchCmdBlockType * blockP)
{
	ZLPluginSampleGlobalType	*globalP = (ZLPluginSampleGlobalType	*)(blockP->globalP);
	ZLPluginLaunchCmdEventType	*cmdEventP = (ZLPluginLaunchCmdEventType	*)(blockP->dataP);
	Err							error = errNone;

	switch(cmdEventP->eventP->eType)
	{
		case penDownEvent:
		{
			if(RctPtInRectangle(cmdEventP->eventP->screenX, cmdEventP->eventP->screenY, &(blockP->clipBounds)))
			{
				ZLPrepareHandlingEvent(blockP); // Call this function before handling an event
				
				if(globalP->x != cmdEventP->eventP->screenX - blockP->bounds.topLeft.x - WIDTH  / 2 ||
				   globalP->y != cmdEventP->eventP->screenY - blockP->bounds.topLeft.y - HEIGHT / 2)
				{
					globalP->x = cmdEventP->eventP->screenX - blockP->bounds.topLeft.x - WIDTH  / 2;
					globalP->y = cmdEventP->eventP->screenY - blockP->bounds.topLeft.y - HEIGHT / 2;
					ZLPluginRefresh(blockP);
				}
				
				ZLShowHint(blockP, blockP->clipBounds.topLeft.x, blockP->clipBounds.topLeft.y);
				
				globalP->boolAllowAnimation	=	false;	//	Disable animation when user is controling the plugin

				cmdEventP->handled = true;
			}
			break;
		}

		case penMoveEvent:
		{
			if(RctPtInRectangle(cmdEventP->eventP->screenX, cmdEventP->eventP->screenY, &(blockP->clipBounds)))
			{
				ZLPrepareHandlingEvent(blockP); // Call this function before handling an event
				
				if(globalP->x != cmdEventP->eventP->screenX - blockP->bounds.topLeft.x - WIDTH  / 2 ||
				   globalP->y != cmdEventP->eventP->screenY - blockP->bounds.topLeft.y - HEIGHT / 2)
				{
					globalP->x = cmdEventP->eventP->screenX - blockP->bounds.topLeft.x - WIDTH  / 2;
					globalP->y = cmdEventP->eventP->screenY - blockP->bounds.topLeft.y - HEIGHT / 2;
					ZLPluginRefresh(blockP);
				}
				
				ZLShowHint(blockP, blockP->clipBounds.topLeft.x, blockP->clipBounds.topLeft.y);
				
				globalP->boolAllowAnimation	=	false;	//	Disable animation when user is controling the plugin

				cmdEventP->handled = true;
			}
			break;
		}

		case penUpEvent:
		{
			if(RctPtInRectangle(cmdEventP->eventP->screenX, cmdEventP->eventP->screenY, &(blockP->clipBounds)))
			{
				ZLPrepareHandlingEvent(blockP); // Call this function before handling an event
				
				ZLHideHint(blockP);

				globalP->boolAllowAnimation	=	true;

				cmdEventP->handled = true;
			}
			break;
		}
		
		
		case nilEvent:	//	Do animation
		{
			if(globalP->boolAllowAnimation)
			{
				if(globalP->xDirection	>	0)
				{
					if(blockP->bounds.topLeft.x + globalP->x	+	WIDTH	>	blockP->clipBounds.topLeft.x	+	blockP->clipBounds.extent.x)
					{
						globalP->xDirection	=	!globalP->xDirection;
						if(globalP->xOffset	==	1)
						{
							globalP->xOffset	=	2;
						}
						else
						{
							globalP->xOffset	=	1;
						}
					}
				}
				else
				{
					if(blockP->bounds.topLeft.x + globalP->x	<	blockP->clipBounds.topLeft.x)
					{
						globalP->xDirection	=	!globalP->xDirection;
						if(globalP->yOffset	==	1)
						{
							globalP->yOffset	=	2;
						}
						else
						{
							globalP->yOffset	=	1;
						}
					}
				}
				if(globalP->yDirection	>	0)
				{
					if(blockP->bounds.topLeft.y + globalP->y	+	HEIGHT	>	blockP->clipBounds.topLeft.y	+	blockP->clipBounds.extent.y)
					{
						globalP->yDirection	=	!globalP->yDirection;
					}
				}
				else
				{
					if(blockP->bounds.topLeft.y + globalP->y	<	blockP->clipBounds.topLeft.y)
					{
						globalP->yDirection	=	!globalP->yDirection;
					}
				}

				if(globalP->xDirection	>	0)
				{
					globalP->x	+=	globalP->xOffset;
				}
				else
				{
					globalP->x	-=	globalP->xOffset;
				}
				if(globalP->yDirection	>	0)
				{
					globalP->y	+=	globalP->yOffset;
				}
				else
				{
					globalP->y	-=	globalP->yOffset;
				}
					
				ZLPluginRefresh(blockP);
			}

			cmdEventP->handled = false;	//	Return false to let the rest installed plugins handles this event too
		
			break;
		}
		
		case frmUpdateEvent:
		{
			//	Do not need to handle frmUpdateEvent, as ZL will calls the plugin's ZLPluginRefresh()
			break;
		}
			
		case ZLDeactivateEvent:
		{
			// DO NOT call ZLPrepareHandlingEvent()
			
			ZLHideHint(blockP);

			globalP->boolAllowAnimation	=	true;

			cmdEventP->handled = true;

			break;
		}
		
		
		case ZLDragAndDropEvent:
		{
			ZLDragAndDropType	*	pZLDragAndDrop;
			UInt32				type;
			
			pZLDragAndDrop	=	(ZLDragAndDropType	*)&(cmdEventP->eventP->data.generic);
			if(pZLDragAndDrop->numItems	>	0	&&	pZLDragAndDrop->item[0].internal)
			{
				if(DmDatabaseInfo(pZLDragAndDrop->item[0].location.internal.cardNo,
									pZLDragAndDrop->item[0].location.internal.dbID,
									0, 0,	0, 0,	0, 0,	0, 0,	0, &type,	0)	==	errNone	&&
					type	==	sysFileTApplication)	//	it's an app
				{
					globalP->cardNo	=	pZLDragAndDrop->item[0].location.internal.cardNo;
					globalP->dbID	=	pZLDragAndDrop->item[0].location.internal.dbID;

					PrvGetAppIcon(blockP);
					PrvGetAppIconName(blockP);
					
					PrvSetHint(blockP);
					
					ZLPluginRefresh(blockP);
				}
			}
			
			cmdEventP->handled = true;
			break;
		}
		
	}

	return error;
}


static Err ZLPluginOpen(ZLPluginLaunchCmdBlockType * blockP)
{
	ZLPluginSampleGlobalType	*globalP;
	UInt32						creator;
	
	//	Initialize the global used by the plugin
	//
	blockP->globalP = MemPtrNew(sizeof(ZLPluginSampleGlobalType));
	if(blockP->globalP == 0)
	{
		return memErrNotEnoughSpace;
	}
	
	globalP = (ZLPluginSampleGlobalType	*)(blockP->globalP);
	MemSet(globalP, sizeof(ZLPluginSampleGlobalType), 0);
	
	globalP->color.r = (UInt8)ZLPrefsReadValue(blockP, PrefsKeyIDColorRedBits, 0);
	globalP->color.g = (UInt8)ZLPrefsReadValue(blockP, PrefsKeyIDColorGreenBits, 0);
	globalP->color.b = (UInt8)ZLPrefsReadValue(blockP, PrefsKeyIDColorBlueBits, 0);
	globalP->color.index = WinRGBToIndex(&globalP->color);
	
	creator = ZLPrefsReadValue(blockP, PrefsKeyIDCreator, 'ZLCR');
	if(creator	!=	0)
	{
		DmSearchStateType stateInfo;
  
		DmGetNextDatabaseByTypeCreator(true, &stateInfo, sysFileTApplication, creator, true,
										&(globalP->cardNo), &(globalP->dbID));
	}

	globalP->iconName[0]	=	0;
	globalP->bitmapP		=	0;

	globalP->xDirection		=	1;
	globalP->yDirection		=	1;

	globalP->xOffset		=	1;
	globalP->yOffset		=	1;

	globalP->boolAllowAnimation	=	true;
	
	PrvGetAppIcon(blockP);
	PrvGetAppIconName(blockP);

	PrvSetHint(blockP);

	return errNone;
}

static Err ZLPluginPreferences(ZLPluginLaunchCmdBlockType * blockP)
{
	ZLPluginSampleGlobalType	*globalP;
	IndexedColorType index;

	globalP = (ZLPluginSampleGlobalType	*)(blockP->globalP);
	if(globalP	!=	0)
	{
		index = globalP->color.index;
		if(UIPickColor (&index, 0, UIPickColorStartPalette, "Select Text Color", 0))
		{
			WinIndexToRGB(index, &(globalP->color));
			globalP->color.index = index;
		}
	}
	return errNone;
}

static Err ZLPluginQuery(ZLPluginLaunchCmdBlockType * blockP)
{
	ZLPluginLaunchCmdQueryType	*queryP = (ZLPluginLaunchCmdQueryType	*)(blockP->dataP);
	Err 						error = errNone;
	
	queryP->supportedZLPluginMgrVerNum = ZLPluginMgrVerNum;	//	should be ZLPluginMgrVerNum
	queryP->updateInterval	=	0;	//	the event update time-out(unit: Tick).
									//	in order to do fast animation, we expect to do an animation every 4 ticks

	//	Return the bounds of the room requested by the plugin
	switch(blockP->position)
	{
		case ZLPluginPosTab:	//	On the plugin tab
			queryP->minWidth	=	30;		//	The minimum width  expected by the plugin.
											//	If the available room is smaller than the minimum requirement,
											//	the plugin will not be initialized
			queryP->minHeight	=	30;		//	The minimum height expected by the plugin.
											//	If the available room is smaller than the minimum requirement,
											//	the plugin will not be initialized
			queryP->maxWidth	=	100;	//	The maximum width  expected by the plugin.
											//	If the available room is larger than the maximum requirement,
											//	only the maximum required room will be available for the plugin.
			queryP->maxHeight	=	100;	//	The maximum height expected by the plugin.
											//	If the available room is larger than the maximum requirement,
											//	only the maximum required room will be available for the plugin.
			break;

		case ZLPluginPosPanel:	//	On the plugin panel
			queryP->minWidth	=	40;		//	The minimum width  expected by the plugin.
											//	If the available room is smaller than the minimum requirement,
											//	the plugin will not be initialized
			queryP->minHeight	=	40;		//	The minimum height expected by the plugin.
											//	If the available room is smaller than the minimum requirement,
											//	the plugin will not be initialized
			queryP->maxWidth	=	blockP->ZLPluginPanelWidth;	//	The maximum width  expected by the plugin.
																//	If the available room is larger than the maximum requirement,
																//	only the maximum required room will be available for the plugin.
			queryP->maxHeight	=	blockP->ZLPluginPanelHeight;//	The maximum height expected by the plugin.
																//	If the available room is larger than the maximum requirement,
																//	only the maximum required room will be available for the plugin.
			break;

		case ZLPluginPosButton:	//	On the Top/Bottom Info Panel
			queryP->minWidth	=	30;		//	The minimum width  expected by the plugin.
											//	If the available room is smaller than the minimum requirement,
											//	the plugin will not be initialized
			queryP->minHeight	=	10;		//	The minimum height expected by the plugin.
											//	If the available room is smaller than the minimum requirement,
											//	the plugin will not be initialized
			queryP->maxWidth	=	40;		//	The maximum width  expected by the plugin.
											//	If the available room is larger than the maximum requirement,
											//	only the maximum required room will be available for the plugin.
			queryP->maxHeight	=	15;		//	The maximum height expected by the plugin.
											//	If the available room is larger than the maximum requirement,
											//	only the maximum required room will be available for the plugin.
			break;
	}
	
	return error;
}

/*
 * FUNCTION: ZLPluginRefresh
 *
 * DESCRIPTION:
 *
 *
 * PARAMETERS:
 *
 *
 * RETURNED:
 */
static Err ZLPluginRefresh(ZLPluginLaunchCmdBlockType * blockP)
{
	ZLPluginSampleGlobalType	*globalP;
	RectangleType				bounds;
	Err							error = errNone;
	FontID						oldFont;
	IndexedColorType			oldTextColor;
	
	globalP = (ZLPluginSampleGlobalType	*)(blockP->globalP);
	
	ZLBeginDraw(blockP, true);	//	Set true to lock the screen before doing custom draw the avoid flicker
								//	Note, locking and unlocking the screen may cause the whole screen flicker
								//	while the program is running on 256 color mode on T|T and T|T2.

	ZLRefreshBackground(blockP);

	oldFont = FntSetFont(stdFont);
	
	oldTextColor = WinSetTextColor(globalP->color.index);


	switch(blockP->position)
	{
		case ZLPluginPosTab:	//	the plugin is on the Plugin Tab
			ZLDrawChars(blockP, globalP->iconName, blockP->bounds.topLeft.x, blockP->bounds.topLeft.y, 0);
			if(globalP->bitmapP	!=	0)
			{
				ZLDrawBitmap(blockP, globalP->bitmapP, blockP->bounds.topLeft.x + globalP->x, blockP->bounds.topLeft.y + globalP->y);
			}
			else
			{
				RctSetRectangle(&bounds, blockP->bounds.topLeft.x + globalP->x, blockP->bounds.topLeft.y + globalP->y, WIDTH, HEIGHT);
				WinInvertRectangleFrame(roundFrame, &bounds);
			}
			break;

		case ZLPluginPosPanel:	//	the plugin is on the Plugin Panel
			ZLDrawChars(blockP, globalP->iconName, blockP->bounds.topLeft.x, blockP->bounds.topLeft.y, 0);
			if(globalP->bitmapP	!=	0)
			{
				ZLDrawBitmap(blockP, globalP->bitmapP, blockP->bounds.topLeft.x + globalP->x, blockP->bounds.topLeft.y + globalP->y);
			}
			else
			{
				RctSetRectangle(&bounds, blockP->bounds.topLeft.x + globalP->x, blockP->bounds.topLeft.y + globalP->y, WIDTH, HEIGHT);
				WinInvertRectangleFrame(roundFrame, &bounds);
			}
			break;

		case ZLPluginPosButton:	//	the plugin is on the Top/Bottom Info Panel
			StrPrintF(globalP->iconName,	"(%d, %d)",	globalP->x, globalP->y);
			ZLDrawChars(blockP, globalP->iconName, blockP->bounds.topLeft.x, blockP->bounds.topLeft.y, 0);
			break;
	}

	WinSetTextColor(oldTextColor);

	FntSetFont(oldFont);

	ZLEndDraw(blockP);

	return error;
}

/*
 * FUNCTION: PilotMain
 *
 * DESCRIPTION: This is the main entry point for the plugin.
 * 
 * PARAMETERS:
 *
 * cmd
 *     word value specifying the launch code. 
 *
 * cmdPB
 *     pointer to a structure that is associated with the launch code
 *
 * launchFlags
 *     word value providing extra information about the launch.
 *
 * RETURNED:
 *     Result of launch, errNone if all went OK
 */
 
UInt32 PilotMain(UInt16 cmd, MemPtr cmdPBP, UInt16 launchFlags)
{
	Err error = ZLPluginErrNotSupportedCmd;
	
	if(cmd	==	sysAppLaunchCmdZLPluginLoad)
	{
		ZLPluginFuncPtr * aZLPluginFuncPtrs	=	(ZLPluginFuncPtr	*)cmdPBP;
		
		aZLPluginFuncPtrs[ZLPluginFuncPtrAbout]			=	ZLPluginAbout;
		aZLPluginFuncPtrs[ZLPluginFuncPtrClose]			=	ZLPluginClose;
		aZLPluginFuncPtrs[ZLPluginFuncPtrHandleEvent]	=	ZLPluginHandleEvent;
		aZLPluginFuncPtrs[ZLPluginFuncPtrOpen]			=	ZLPluginOpen;
		aZLPluginFuncPtrs[ZLPluginFuncPtrPreferences]	=	ZLPluginPreferences;
		aZLPluginFuncPtrs[ZLPluginFuncPtrQuery]			=	ZLPluginQuery;
		aZLPluginFuncPtrs[ZLPluginFuncPtrRefresh]		=	ZLPluginRefresh;
		
		
		error	=	errNone;
	}

	return error;
}

