/*
 * asmloader.c
 *
 * Linux Loader for the Palm OS.
 *
 * Copyright (C) 2004 Vinayak R. Borkar
 * Author: Vinayak R. Borkar [vinayakb@users.sourceforge.net]
 *
 * Original version by Romain Goyet [r.goyet@gmail.com]
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <PceNativeCall.h>
#include <Standalone.h>

#include "garux.h"

#define SET_TAG_WORD(off, value) *((unsigned long *)(0x10000200+((off)<<2)))=value
#define SET_TAG_WORD_BYTE(off, byte, value) *((char *)(0x10000200+((off)<<2)+(byte)))=value

STANDALONE_CODE_RESOURCE_ID (0);

#define DEBUG(a) { \
    unsigned long register __temp__; \
    for(__temp__=0 ; __temp__<1024 ; __temp__+=4) { \
        *((unsigned long *)((a)+__temp__))=0xffffffff; \
    } \
}

void startKernel();

unsigned long bootLinux(const void *emulStateP, char *userData68KP, Call68KFuncType *call68KFuncP) {
    unsigned long register i;
    unsigned long register j;
    unsigned long register krnlSrc;
    
    krnlSrc=(unsigned long)userData68KP;

    // Disable FIQ, IRQ
    asm volatile ("mrs r0, cpsr");
    asm volatile ("orr r0, r0, #0xc0");
    asm volatile ("msr cpsr, r0");

    // Set Page table entries for 0x10000000 and 0x11000000 to identity.
    *((unsigned long *)0x4400)=0x10000c1a;
    *((unsigned long *)0x4440)=0x11000c1a;

    // Let's flush Instruction and Data cache
    asm volatile ("mov r0, #0");
    asm volatile ("mcr p15, 0, r0, c7, c7, 0");       // flush v3/v4 cache
    asm volatile ("mcr p15, 0, r0, c8, c7, 0");       // flush v4 TLB

    // Wait a bit.
    for(i=0 ; i<100000 ; i++);

    DEBUG(0x11f08000);

    // Copy the kernel to memory starting at 0x11000000
    // In reality we want to copy the image to 0x10008000, but that location
    // holds a second level page table. So if we try to copy directly to the
    // final destination, the system crashes due to a corrupted second level
    // page table. So we temporarily copy it way out into 0x11000000 first.
    // 
    // Assumption: 0x11000000 is way out and does not trample on either
    // the code or kernel data.
    for(i=0 ; i<NUM_KERNEL_PAGES*PAGE_SIZE ; i+=4) {
        *((unsigned long *)(0x11000000+i))=*((unsigned long *)(krnlSrc+i));
    }

    // Set the ATags

    i=0;
    // ATAG_CORE
    SET_TAG_WORD(i++,0x00000002);
    SET_TAG_WORD(i++,0x54410001);

    // ATAG_MEM
    SET_TAG_WORD(i++,0x00000004);
    SET_TAG_WORD(i++,0x54410002);
    SET_TAG_WORD(i++,0x02000000);
    SET_TAG_WORD(i++,0x10000000);

    // ATAG_CMDLINE
    SET_TAG_WORD(i++,0x00000007);
    SET_TAG_WORD(i++,0x54410009);
    SET_TAG_WORD_BYTE(i,0,'c');
    SET_TAG_WORD_BYTE(i,1,'o');
    SET_TAG_WORD_BYTE(i,2,'n');
    SET_TAG_WORD_BYTE(i++,3,'s');
    SET_TAG_WORD_BYTE(i,0,'o');
    SET_TAG_WORD_BYTE(i,1,'l');
    SET_TAG_WORD_BYTE(i,2,'e');
    SET_TAG_WORD_BYTE(i++,3,'=');
    SET_TAG_WORD_BYTE(i,0,'/');
    SET_TAG_WORD_BYTE(i,1,'d');
    SET_TAG_WORD_BYTE(i,2,'e');
    SET_TAG_WORD_BYTE(i++,3,'v');
    SET_TAG_WORD_BYTE(i,0,'/');
    SET_TAG_WORD_BYTE(i,1,'t');
    SET_TAG_WORD_BYTE(i,2,'t');
    SET_TAG_WORD_BYTE(i++,3,'y');
    SET_TAG_WORD_BYTE(i,0,'S');
    SET_TAG_WORD_BYTE(i,1,'2');
    SET_TAG_WORD_BYTE(i,2,0);
    SET_TAG_WORD_BYTE(i++,3,0);

    // ATAG_OMAP
    SET_TAG_WORD(i++,0x00000005);
    SET_TAG_WORD(i++,0x414f4d50);
    SET_TAG_WORD(i++,0x4f030005);
    SET_TAG_WORD(i++,0x030001c2);
    SET_TAG_WORD(i++,0x00000000);
   
    // ATAG_NONE
    SET_TAG_WORD(i++,0x00000000);
    SET_TAG_WORD(i++,0x00000000);

    // FIXME: We arbitrarily add 0x60 to pc and copy 512 bytes starting
    // from there. So if any code is modified between <CRITICAL>
    // and </CRITICAL>, the offset needs to change.
    // <CRITICAL>
    asm volatile ("add %0, pc, #0x60" : "=r" (j));
    for(i=0 ; i<512 ; i++) {
        *((char *)(0x10000000+i))=*((char *)(i+j));
    }

    DEBUG(0x11f09000);

    asm volatile ("mov pc, #0x10000000");

    // </CRITICAL>

    asm volatile ("nop");
    asm volatile ("nop");
    asm volatile ("nop");
    asm volatile ("nop");
    asm volatile ("nop");
    asm volatile ("nop");
    asm volatile ("nop");

    // Let's flush Instruction and Data cache
    asm volatile ("mov r0, #0");
    asm volatile ("mcr p15, 0, r0, c7, c7, 0");       // flush v3/v4 cache
    asm volatile ("mcr p15, 0, r0, c8, c7, 0");       // flush v4 TLB

    // Wait a bit.
    for(i=0 ; i<100000 ; i++);
    
    // Let's disable the MMU and disable data cache.
    asm volatile ("mrc p15, 0, r0, c1, c0, 0");
    asm volatile ("bic r0, r0, #0x00002300");         // clear bits 13, 9:8 (--V- --RS)
    asm volatile ("bic r0, r0, #0x00000087");         // clear bits 7, 2:0 (B--- -CAM)
    asm volatile ("orr r0, r0, #0x00001000");         // set bit 12 (I) I-Cache
    asm volatile ("mcr p15, 0, r0, c1, c0, 0");
    
    // Wait a bit.
    for(i=0 ; i<100000 ; i++);
    
    DEBUG(0x11f0a000);
    
    // Actually copy the kernel to 0x10008000.
    for(i=0 ; i<NUM_KERNEL_PAGES*PAGE_SIZE ; i+=4) {
        *((unsigned long *)(0x10008000+i))=*((unsigned long *)(0x11000000+i));
    }

    DEBUG(0x11f0b000);

    // Wait a bit.
    for(i=0 ; i<10000000 ; i++);

    // Initializing before boot
    asm volatile ("mov r0, #0");
    asm volatile ("mov r1, #452");
    asm volatile ("mov r2, #0x10000000");
    asm volatile ("add r2, r2, #0x200");

    // Jump to the kernel.
    asm volatile ("mov r3, #0x8000");
    asm volatile ("orr r3, r3, #0x10000000");
    asm volatile ("mov pc, r3");

    // We should never get here... Just to make the compiler happy.
    return 0;
}
