// Copyright (C) 2004-6 Alexander R. Pruss

#include <PalmOS.h>
#include "ForceDelete_Res.h"

#define CHECKMARK_SIZE 16


typedef struct {
    Char name[ dmDBNameLength ];
    UInt16  card;
    LocalID id;
    Boolean selected;
} FileInfo;


typedef struct {
    UInt16 launchCode;
    MemHandle panelIDsH;
    SysDBListItemType* panelData;
    Char** panelNames;
    UInt16 panelCount;
    UInt16 numFiles;
    FileInfo** files;
    Char** fileNames;
} ForceDeleteGlobals;



void FatalError( UInt16 alert )
{
     FrmAlert( alert );
     ErrThrow( 1 );
}


void SafeFtrPtrFree( UInt16 feature )
{
    UInt32 value;

    if ( errNone != FtrGet( appCreator, feature, &value ) )
        return;
        
    FtrPtrFree( appCreator, feature );
}


void* SafeMemPtrNew( UInt32 size )
{
    void* p;

    p = MemPtrNew( size );

    if ( p == NULL ) {
        FatalError( altNoMem );
    }

    return p;
}


void SafeMemPtrFree( void* p )
{
    if ( p != NULL )
        MemPtrFree( p );
}


UInt16 GetObjectIndex( UInt16 id )
{
    return FrmGetObjectIndex( FrmGetActiveForm(), id );
}


void* GetObjectPtr( UInt16 id )
{
    return FrmGetObjectPtr( FrmGetActiveForm(), GetObjectIndex( id ) );
}



Int16 GetSelection( UInt16 lstID )
{
    return LstGetSelection( GetObjectPtr( lstID ) );
}


void SetList( UInt16 trgID, UInt16 lstID, Int16 sel )
{
    LstSetSelection( GetObjectPtr( lstID ),
        sel );
    CtlSetLabel( GetObjectPtr( trgID ),
        LstGetSelectionText( GetObjectPtr( lstID ),
        sel ) );
}

void SetValue( UInt16 id, UInt16 value )
{
    CtlSetValue( GetObjectPtr( id ), value );
}


UInt16 GetValue( UInt16 id )
{
    return CtlGetValue( GetObjectPtr( id ) );
}


void Show( UInt16 id, Boolean state )
{
    UInt16 index = GetObjectIndex( id );
    FormType* currentForm = FrmGetActiveForm();

    if ( state )
        FrmShowObject( currentForm, index );
    else
        FrmHideObject( currentForm, index );
    if ( FrmGetObjectType( currentForm, index ) == frmControlObj )
        CtlSetUsable( GetObjectPtr( id ), state);
}


void ClearPanelList( ForceDeleteGlobals* g )
{
    if ( 0 < g->panelCount ) {
        MemPtrFree( g->panelNames );
        MemHandleUnlock( g->panelIDsH );
        MemHandleFree( g->panelIDsH );

        g->panelNames = NULL;
        g->panelCount = 0;
    }
}



void FixObjectWidthRight( UInt16 id, Coord width )
{
    RectangleType bounds;

    FrmGetObjectBounds( FrmGetActiveForm(), GetObjectIndex( id ), &bounds );

    bounds.topLeft.x = bounds.topLeft.x + bounds.extent.x - width;
    bounds.extent.x = width;

    FrmSetObjectBounds( FrmGetActiveForm(), GetObjectIndex( id ), &bounds );
}



static Boolean InROM( UInt16 card, LocalID id )
{
    return MemLocalIDKind( id ) == memIDPtr;
}



void InitPanelList( ForceDeleteGlobals* g )
{
    Int16     myPanel;
    Coord     maxWidth;

    ClearPanelList( g );

    myPanel  = -1;
    maxWidth = 0;

    if ( SysCreatePanelList( &g->panelCount, &g->panelIDsH ) &&
         0 < g->panelCount ) {
        g->panelData = MemHandleLock( g->panelIDsH );

        g->panelNames = MemPtrNew( g->panelCount * sizeof( Char* ) );

        if ( g->panelNames == NULL ) {
            g->panelCount = 0;
            MemHandleUnlock( g->panelIDsH );
            MemHandleFree( g->panelIDsH );
        }
        else {
            UInt16 i;
            UInt16 card;
            LocalID id;
            Err    err;

            FntSetFont( stdFont );

            err = SysCurAppDatabase( &card, &id );

            if ( err != errNone ) {
                card = 0xFFFF;
                id = NULL;
            }

            for ( i = 0 ; i < g->panelCount ; i++ ) {
                 Coord w;

                 g->panelNames[ i ] = g->panelData[ i ].name;
                 if ( g->panelData[ i ].dbID == id )
                     myPanel = i;
                 w = FntCharsWidth( g->panelNames[ i ], StrLen( g->panelNames[ i ] ) );

                 if ( maxWidth < w )
                     maxWidth = w;
            }
        }
    }

    if ( 0 < g->panelCount ) {
        Coord selectionWidth;

        selectionWidth = 18;

        if ( 0 <= myPanel )
            selectionWidth += FntCharsWidth( g->panelNames[ myPanel ],
                                  StrLen( g->panelNames[ myPanel ] ) );

        LstSetListChoices( GetObjectPtr( lstPanelList ), g->panelNames, g->panelCount );
        LstSetSelection( GetObjectPtr( lstPanelList ), myPanel );
        LstSetHeight( GetObjectPtr( lstPanelList ), g->panelCount );
        FixObjectWidthRight( trgPanelList, selectionWidth );
        FixObjectWidthRight( lstPanelList, maxWidth );
    }
    else {
        LstSetListChoices( GetObjectPtr( lstPanelList ), NULL, 0 );
    }
    if ( 0 <= myPanel ) {
        CtlSetLabel( GetObjectPtr( trgPanelList ), g->panelNames[ myPanel ] );
    }
    else {
        CtlSetLabel( GetObjectPtr( trgPanelList ), "" );
    }
}



void PreparePanel( ForceDeleteGlobals* g )
{
    if ( g->launchCode == sysAppLaunchCmdPanelCalledFromApp ) {
        Show( btnDone, true );
        Show( trgPanelList, false );
        Show( lblTitle, true );
    }
    else {
        InitPanelList( g );

        Show( btnDone, false );
        Show( trgPanelList, true );
        Show( lblTitle, false );
    }

}


static Boolean HandleMenu( UInt16 id )
{
    Boolean handled;

    handled = false;

    switch( id ) {
        case mnuAbout: {
            FrmPopupForm( frmAbout );
            handled = true;
            break;
        }

        default:
            break;
     }

     return handled;
}


static Boolean Handler_frmAbout( EventType* event )
{
    Boolean handled;

    handled = false;

    switch (event->eType) {
        case frmOpenEvent: {
            FrmDrawForm( FrmGetActiveForm() );
            handled = true;
            break;
        }

        case ctlSelectEvent: {
            switch( event->data.ctlEnter.controlID ) {
                case btnOK:
                    FrmReturnToForm( 0 );
                    handled = true;
                    break;
            }
            break;
        }

        default:
            break;
    }
    return handled;
}


static void DrawFile( Int16 itemNum, RectangleType* bounds, Char** itemsText )
{
    FontID oldFont;
    ForceDeleteGlobals* g;

    FtrGet( appCreator, FTR_GLOBALS, (UInt32*)&g );

    oldFont = FntSetFont( symbol11Font );

    WinDrawChar( !! g->files[ itemNum ]->selected, bounds->topLeft.x, bounds->topLeft.y );

    FntSetFont( oldFont );

    WinDrawChars( itemsText[ itemNum ], StrLen( itemsText[ itemNum ] ),
        bounds->topLeft.x + CHECKMARK_SIZE, bounds->topLeft.y );
}



Boolean GetNextFile( Boolean first, DmSearchStateType* state,
    UInt16* cardP, LocalID* dbIDP, Char* name )
{
    while ( errNone == DmGetNextDatabaseByTypeCreator( first, state,
               0, 0, false, cardP, dbIDP ) ) {
        first = false;
        if ( InROM( *cardP, *dbIDP ) )
            continue;
        if ( name == NULL )
            return true;
        if ( errNone ==
             DmDatabaseInfo( *cardP, *dbIDP, name, NULL, NULL,
                 NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL ) )
            return true;
    }
    return false;
}


static void ClearList( ForceDeleteGlobals* g )
{
    if ( g->files != NULL ) {
        UInt16 i;
        for ( i = 0 ; i < g->numFiles ; i++ )
             SafeMemPtrFree( g->files[ i ] );
        MemPtrFree( g->files );
        g->files = NULL;
        g->numFiles = 0;
    }

    if ( g->fileNames != NULL ) {
        MemPtrFree( g->fileNames );
        g->fileNames = NULL;
    }

    g->numFiles = 0;
}


Int16 CompareFiles( void* a, void* b, Int32 discard )
{
    return StrCaselessCompare( ( *( FileInfo** )a )->name, ( *( FileInfo** )b )->name );
}


static void LoadList( ForceDeleteGlobals* g )
{
    UInt16 count;
    UInt16 i;
    DmSearchStateType state;
    UInt16 card;
    LocalID id;
    ListType* list;

    ClearList( g );

    list = GetObjectPtr( lstFiles );

    LstSetListChoices( list, NULL, 0 );

    for ( count = 0 ; GetNextFile( count == 0, &state, &card, &id, NULL ) ; count++ );

    if ( 0 < count ) {
        g->files     = SafeMemPtrNew( sizeof( FileInfo* ) * count );
        MemSet( g->files, sizeof( FileInfo* ) * count, 0 );
        g->fileNames = SafeMemPtrNew( sizeof( Char* ) * count );

        for ( i = 0 ; i < count ; i++ ) {
             UInt16 card;

             g->files[ i ] = SafeMemPtrNew( sizeof( FileInfo ) );
             MemSet( g->files[ i ], sizeof( FileInfo ), 0 );

             if ( ! GetNextFile( i == 0, &state, &( g->files[ i ]->card ), &( g->files[ i ]->id ),
                 g->files[ i ]->name ) ) {
                 g->files[ i ] = NULL;
                 MemPtrFree( g->files[ i ] );
                 break;
             }
        }

        g->numFiles = i;

        SysInsertionSort( g->files, g->numFiles, sizeof( FileInfo* ), CompareFiles, 0 );

        for ( i = 0 ; i < g->numFiles ; i++ ) {
             g->fileNames[ i ] = g->files[ i ]->name;
        }
    }

    LstSetListChoices( list, g->fileNames, g->numFiles );
    LstSetSelection( list, -1 );
    LstSetDrawFunction( list, DrawFile );
    LstDrawList( list );
}




void SetRecyclable( UInt16 card, LocalID id )
{
    UInt16 attr;

    if ( id == NULL )
        return;

    if ( errNone == DmDatabaseInfo( card, id, NULL, &attr, NULL,
         NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL ) ) {

        attr = attr | dmHdrAttrRecyclable;
        DmSetDatabaseInfo( card, id, NULL, &attr, NULL,
                  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL );
    }

}


static Boolean HaveNVFS( void )
{
    Err err;
    UInt32 value;

    err = FtrGet( sysFtrCreator, sysFtrNumDmAutoBackup, &value );

    return ( err == errNone && value != 0 );
}



static Coord ItemWidth( Int16 item )
{
    Char* string;

    string = LstGetSelectionText( GetObjectPtr( lstFiles ), item );

    return FntCharsWidth( string, StrLen( string ) );
}


static void ScrollTo( Char c )
{
    Int16 i;
    UInt16 n;
    ListType* list;

    list = GetObjectPtr( lstFiles );

    n = LstGetNumberOfItems( list );

    for ( i = 0 ; i < n ; i++ ) {
         Char* name = LstGetSelectionText( list, i );

         if ( 0 == StrNCaselessCompare( &c, name, 1 ) )
             break;

         if ( StrNCaselessCompare( &c, name, 1 ) < 0 ) {
             i--;
             break;
         }
    }

    if ( i == n )
        i = n - 1;
    if ( i < 0 )
        i = 0;
    LstMakeItemVisible( list, i );
    LstDrawList( list );
}



static Boolean Handler_frmMain( EventType* event )
{
    Boolean handled;

    ForceDeleteGlobals* g;

    FtrGet( appCreator, FTR_GLOBALS, (UInt32*)&g );

    handled = false;

    switch (event->eType) {
        case winEnterEvent:
            handled = false;
            break;

        case frmOpenEvent: {
            LoadList( g );

            PreparePanel( g );

            FrmDrawForm( FrmGetActiveForm() );

            handled = true;
            break;
        }

        case keyDownEvent: {
            if ( event->data.keyDown.modifiers & commandKeyMask ) {
                switch ( event->data.keyDown.chr ) {
                    case vchrRockerUp:
                    case chrPageUp:
                        LstScrollList( GetObjectPtr( lstFiles ), winUp,
                            LstGetVisibleItems( GetObjectPtr( lstFiles ) ) - 1 );
                        handled = true;
                        break;
                    case vchrRockerDown:
                    case chrPageDown:
                        LstScrollList( GetObjectPtr( lstFiles ), winDown,
                            LstGetVisibleItems( GetObjectPtr( lstFiles ) ) - 1 );
                        handled = true;
                        break;
                    default:
                        break;
                }
            }
            else {
                if ( 0 == (0xFF00 & event->data.keyDown.chr ) ) {
                    ScrollTo( event->data.keyDown.chr );
                    handled = true;
                }
            }
            break;
        }

        case lstSelectEvent: {
            Int16 item;
            RectangleType bounds;

            item = event->data.lstSelect.selection;

            FrmGetObjectBounds( FrmGetActiveForm(),
                GetObjectIndex( event->data.lstSelect.listID ), &bounds );

            switch ( event->data.lstSelect.listID ) {
                case lstFiles:
                    if ( 0 <= item &&
                         event->screenX < bounds.topLeft.x + CHECKMARK_SIZE +
                             ItemWidth( item ) ) {
                        g->files[ item ]->selected = ! g->files[ item ]->selected;
                    }
                    LstSetSelection( GetObjectPtr( lstFiles ), -1 );
                    LstDrawList( GetObjectPtr( lstFiles ) );
                    handled = true;
                    break;
            }
            break;
        }

        case popSelectEvent:
            if ( event->data.popSelect.selection != noListSelection ) {
                switch ( event->data.popSelect.controlID ) {
                    case trgPanelList: {
                        SysDBListItemType* item;

                        item = g->panelData + event->data.popSelect.selection;

                        SysUIAppSwitch( item->cardNo, item->dbID, 0, NULL );
                        
                        handled = true;
                        break;
                    }
                }
            }

        case ctlSelectEvent: {
            UInt16       id;
            Int16        sel;

            sel = GetSelection( lstFiles );

            id = event->data.ctlEnter.controlID;

            switch( event->data.ctlEnter.controlID ) {
                case btnGo: {
                    UInt16 i;
                    UInt16 count;

                    count = 0;

                    for ( i = 0 ; i < g->numFiles ; i++ )
                         if ( g->files[ i ]->selected )
                             count++;

                    if ( 0 < count ) {
                        Char string[ 10 ];
                        StrPrintF( string, "%u", count );
                        if ( 0 == FrmCustomAlert( altDelete, string, (count == 1) ? "" : "s", "" ) ) {
                            for ( i = 0 ; i < g->numFiles ; i++ )
                                 if ( g->files[ i ]->selected )
                                     SetRecyclable( g->files[ i ]->card, g->files[ i ]->id );
                            if ( HaveNVFS() )
                                DmSync();
                            SysReset();
                        }
                    }
                    handled = true;
                    break;
                }
                case btnUncheck: {
                    UInt16 i;
                    for ( i = 0 ; i < g->numFiles ; i++ )
                         g->files[ i ]->selected = false;
                    LstDrawList( GetObjectPtr( lstFiles ) );
                    handled = true;
                    break;
                }
                case btnDone: {
                    EventType e;

                    MemSet( &e, sizeof( EventType ), 0 );
                    e.eType = appStopEvent;

                    EvtAddEventToQueue( &e );
                    break;
                }
            }

            break;
        }

        default:
            break;
    }

    return handled;
}



static void EventLoop( ForceDeleteGlobals* g )
{
    EventType event;
    Boolean   handled;

    do {
       	EvtGetEvent( &event, evtWaitForever );

       	handled = SysHandleEvent( &event );

       	if ( ! handled ) {
       	    Err err;

       	    handled = MenuHandleEvent( NULL, &event, &err );
       	}

       	if ( ! handled ) {

            switch ( event.eType ) {
                case frmLoadEvent: {
                    UInt16    formID;
                    FormType* form;

                    formID = event.data.frmLoad.formID;
                    form = FrmInitForm( formID );

                    FrmSetActiveForm( form );

                    switch ( formID ) {
                        case frmMain:
                            FrmSetEventHandler( form, Handler_frmMain );
                            break;
                        case frmAbout:
                            FrmSetEventHandler( form, Handler_frmAbout );
                            break;
                    }
                    handled = true;
                }
                case menuEvent:
                    handled = HandleMenu( event.data.menu.itemID );
                    break;
                default:
                    handled = FrmDispatchEvent( &event );
            }
       	}
    } while ( event.eType != appStopEvent );

    FrmCloseAllForms();
}



static Boolean ApplicationStart( ForceDeleteGlobals* g )
{
    FtrSet( appCreator, FTR_GLOBALS, ( UInt32 )g );
    FrmGotoForm( frmMain );

    return true;
}


static void ApplicationStop( ForceDeleteGlobals* g )
{
    ClearList( g );
    ClearPanelList( g );
}



static void Run( UInt16 command )
{
    Err err;
    ForceDeleteGlobals* g;
    
    g = MemPtrNew( sizeof( ForceDeleteGlobals ) );

    if ( g == NULL )
        return;
        
    MemSet( g, sizeof( ForceDeleteGlobals ), 0 );

    g->launchCode = command;

    ErrTry {
        err = errNone;
        if ( ApplicationStart( g ) )
            EventLoop( g );
        ApplicationStop( g );
    }
    ErrCatch( err ) {
        ApplicationStop( g );
    } ErrEndCatch

    MemPtrFree( g );
}


UInt32 PilotMain( UInt16 command, void* cmdPBP, UInt16 flags )
{
    switch ( command ) {
        case sysAppLaunchCmdPanelCalledFromApp:
        case sysAppLaunchCmdNormalLaunch:
            Run( command );
            break;
#if 0
        case sysAppLaunchCmdSystemReset: {
            UInt32 oldMask;
            Boolean go;

            go = 0 != ( KeyCurrentState() & keyBitHard2 );

            if ( go ) {
                oldMask = KeySetMask( ~(UInt32)keyBitHard2 );
                EvtFlushKeyQueue();
                Run( sysAppLaunchCmdPanelCalledFromApp );
                KeySetMask( oldMask | keyBitHard2 );
            }

            break;
        }
#endif
    }
}

